// IntroMaker for QuakeC by NiKoDeMoS
// midctf@quakemania.com
// http://www.quakemania.com/midctf/


#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <stdio.h>

// various function types
const int       FUNC_START =    1;
const int       FUNC_EFFECT =   2;
const int       FUNC_EFFEND =   3;
const int       FUNC_SHOW =     4;
const int       FUNC_END =      5;

// globals
char            buf[2048], *bptr;   // buffer w/ dynamic pointer
float           show, delay;        // show time, delay between messages
char            messagetxt[2048];   // message text
int             start, end, linenum, ctrls;  // start, end, line number,
                                             // count of controls
// prototypes
void GetCommand(char *c);

// array of pointers to functions
struct func {
        int     functiontype;
        char    funcname[10];
        float   show, pause;
        char    text[2048];
        }       *funcstack[2048];

int     cur_func;                       // current function

// generate error text (message *s at line number num)
void Error(int num, char *s)
{
        if (num) num = linenum;
        cout << "Error: ";
        cout << num;
        cout << ": ";
        cout << s;
        cout << "\n";
        exit(1);
}

// copy src constant into dest string
void strcpy(char *dest, const char *src)
{
        while (*src)
                *dest++ = *src++;
        *dest = (char)NULL;
}

// compare two strings
int strcmp(char *doc, const char *tmp)
{
        while (*tmp)
                if (*tmp++ != *doc++) return 1;
        return 0;
}

// return the length of a string
int strlen(char *s)
{
        int x;
        x = 0;
        while (*s++)
                x++;
        return x;
}

// create space for a function
struct func *CreateFunc ()
{
        funcstack[cur_func] = (struct func *)malloc(sizeof(funcstack));
        return funcstack[cur_func++];
}

// return the value of a digit
float Digits (char ch)
{
        switch (ch) {
                case '0':       return 0;
                case '1':       return 1;
                case '2':       return 2;
                case '3':       return 3;
                case '4':       return 4;
                case '5':       return 5;
                case '6':       return 6;
                case '7':       return 7;
                case '8':       return 8;
                case '9':       return 9;
                default:        Error(1, "bad digit");
      }
}

// parse a number from the active buffer
void GetANum (float *num)
{
        float decimal;
        float pro_num;
        pro_num = 0;
        decimal = 0;
        if (!*bptr) return;  // end of buffer
        while (*bptr && *bptr == ' ')   // roll through whitespace
                *bptr++;
        if (*bptr < '0' || *bptr > '9')  // no number, return
                return;
        while (*bptr && *bptr != ' ' && *bptr != '\n')  // roll through #
        {
                if (*bptr == '.') // process decimal
                {
                        if (decimal)
                                Error(1, "bad decimal");
                        else    decimal = 10;
                }
                else if (*bptr < '0' || *bptr > '9') // not a digit!
                        Error (1, "funky number");
                else if (decimal)
                {
                        pro_num += Digits(*bptr) / decimal; // add decimal
                        decimal *= 10;
                }
                else
                {
                        pro_num *= 10;              // add whole part
                        pro_num += Digits(*bptr);
                }
                *bptr++;
        }
        *num = pro_num;
}

// parse a string from the active buffer
void GetAString (char *s)
{
        *s = (char)NULL;   // null terminate
        if (!*bptr) return;   // end of buffer
        while (*bptr == ' ')  // roll through white space
                *bptr++;
        if (*bptr++ != '\"')   // expect a "
                Error(1, "expected string");
        while (*bptr && *bptr != '\n')   // roll through string
        {
                if (*bptr == '\"')  // terminate with "
                {
                        *s = (char)NULL;
                        return;
                }
                if (*bptr == '\\')    // process \ special character
                        *s++ = *bptr++;
                if (*bptr)
                        *s++ = *bptr++;
        }
        Error(1, "non-terminated string"); // we got out w/out a closing "
} 
                
// create a start function
void Start ()
{
        if (start)
                Error(1, "multiple start tokens");
        start = 1;       
        struct func *work;
        GetANum(&show);
        GetANum(&delay);
        work = CreateFunc ();
        work->functiontype = FUNC_START;
        strcpy(work->text, messagetxt);
        work->show = show;
        work->pause = delay;
}

// create a display function
void Show ()
{
        struct func *work;
        if (!start)
                Error(1, "no start");
        GetANum(&show);
        GetANum(&delay);
        work = CreateFunc ();
        work->functiontype = FUNC_SHOW;
        work->show = show;
        work->pause = delay;
        strcpy(work->text, messagetxt);
}

void Pan (int up, int out)
{
        int total, step, x, y, z;
        float timestep, fframes, ftime;
        char *p, *q;
        struct func *work;
        GetANum(&fframes);
        GetANum(&ftime);
        if (fframes < 2.0) Error (1, "bad pan framecount");
        if (fframes >= strlen(messagetxt)) Error (1, "bad pan framecount");
        if (ftime < 0.1) Error (1, "bad pan time");
        total = strlen(messagetxt);
        step = (int)((float)total / fframes);
        timestep = ftime / (fframes - 1.0);
        if (up)
                x = 0;
        else
                x = total;
        while (up ? x < total : x > 0)
        {
                work = CreateFunc ();
                work->functiontype = FUNC_EFFECT;
                work->pause = timestep;
                if (out)
                {
                        y = 0;
                        p = work->text;
                        while (y++ < x)
                                *p++ = ' ';
                        q = messagetxt;
                        while (y++ < total && *q)
                                *p++ = *q++;
                        *p = (char)NULL; 
                }
                else
                {
                        y = 0;
                        z = x;
                        q = messagetxt;
                        while (*q) q++;
                        while (z-- > 0)
                                q--;
                        p = work->text;
                        while (*q)
                        {
                                y++;
                                *p++ = *q++;
                        }
                        while (y++ < total)
                                *p++ = ' ';
                        *p = (char)NULL;
                }
                if (up)
                        x += step;
                else
                        x -= step;
        }
        work->functiontype = FUNC_EFFEND;
        work->show = show;
        work->pause = delay;
}
                           
void Pan ()
{
        char dir[20], move[20];
        GetCommand(dir);
        GetCommand(move);
        if (strcmp(dir, "L") && strcmp(dir, "R")) Error(1, "bad pan direction");
        if (strcmp(move, "I") && strcmp(move, "O")) Error(1, "bad pan type");
        if (!strcmp(dir, "L") && !strcmp(move, "I")) Pan (0, 1);
        else if (!strcmp(dir, "R") && !strcmp(move, "O")) Pan (1, 1);
        else if (!strcmp(dir, "L") && !strcmp(move, "O")) Pan (0, 0);
        else Pan (1, 0); 
};

void Plosion (int dir)
{
        struct func *work;
        float frames, time, tstep;
        int x, y, h, xstep, total;
        char *p, *q, *r;
        GetANum(&frames);
        GetANum(&time);
        total = strlen(messagetxt);
        total = total / 2;
        if ((int)frames >= total) Error(1, "bad plosion framecount");
        xstep = (int)((float)total / frames);
        tstep = time / ((float)total - 1.0);
        if (dir)
                x = 0;
        else
                x = total;
        while (dir ? x < total : x >= 0)
        {
                work = CreateFunc ();
                work->functiontype = FUNC_EFFECT;
                work->pause = tstep;
                p = work->text;
                q = messagetxt;
                r = messagetxt;
                while (*r) r++;
                y = total - x;
                while (y-- > 0)
                {
                        *p++ = *q++;
                        r--;
                }
                y = total - x;
                while (y-- > 0)
                        *p++ = *r++;
                *p = (char)NULL;
                if (dir)
                        x += xstep;
                else
                        x -= xstep;
        }
        work->functiontype = FUNC_EFFEND;
        work->show = show;
}

        
                        
// create an ending function
void End ()
{
        struct func *work;
        if (!start)
                Error(1, "end before start");
        if (end)
                Error(1, "multiple end tokens");
        else    end = 1;
        GetANum(&delay);
        work = CreateFunc ();
        work->functiontype = FUNC_END;
        work->pause = delay;
        strcpy(work->text, messagetxt);
}

// parse the next command from the active buffer 
void GetCommand (char *command)
{
        char *p;
        p = bptr;
        while (*p == ' ') *p++;
        while (*p && *p != ' ' && *p != '\n')
                *command++ = *p++;
        *command = (char)NULL;
        bptr = p;
}
               
// process the data in the current buffer
void ProcessBuffer()
{
        char command[20], errormsg[40];
        bptr = buf + 1;
        GetCommand (command);
        if (!strcmp(command, "start")) Start ();
        else if (!strcmp(command, "end")) End ();
        else if (!strcmp(command, "delay")) GetANum (&delay);
        else if (!strcmp(command, "showfor")) GetANum (&show);
        else if (!strcmp(command, "settext")) GetAString (messagetxt);
        else if (!strcmp(command, "showtext")) Show ();
        else if (!strcmp(command, "pan")) Pan ();
        else if (!strcmp(command, "implode")) Plosion (1);
        else if (!strcmp(command, "explode")) Plosion (0);
        else {
                sprintf(errormsg, "%s: bad command", command);
                Error(1, errormsg);
        }
        ctrls++;
}

// parse the functions and build source into the buffer
void MakeOutputBuffer ()
{
        struct func *work;
        cur_func--;
        work = (struct func *)funcstack[cur_func];
        if (work->functiontype == FUNC_START)
                sprintf(buf,"void() Intro = {\n     local entity intro;\n     intro = spawn ();\n     intro.owner = self;\n     intro.think = Intro1;\n     intro.attack_finished = time + %f;\n     intro.nextthink = time + %f;\n};\n\n",work->show + work->pause, work->pause);
        else if (work->functiontype == FUNC_EFFECT)
                sprintf(buf,"void() Intro%d = {\n     centerprint(self.owner, \"%s\");\n     self.think = Intro%d;\n     self.nextthink = time + %f;\n};\n\n", cur_func, work->text, cur_func + 1, work->pause);
        else if (work->functiontype == FUNC_EFFEND)
                sprintf(buf,"void() Intro%d = {\n     centerprint(self.owner, \"%s\");\n     self.attack_finished = time + %f;\n     self.think = Intro%d;\n     self.nextthink = time + %f;\n};\n\n", cur_func, work->text, work->show + work->pause, cur_func + 1, work->pause);
        else if (work->functiontype == FUNC_END)
                sprintf(buf,"void() Intro%d = {\n     if (time > self.attack_finished) {\n          centerprint(self.owner, \"\");\n          self.think = SUB_Remove;\n          self.nextthink = time + %f;\n     }\n     else {\n          centerprint (self.owner, \"%s\");\n          self.nextthink = time + 0.1;\n     }\n};\n\n",cur_func,work->pause,work->text);
        else    sprintf(buf,"void() Intro%d = {\n     if (time > self.attack_finished) {\n          self.think = Intro%d;\n          self.nextthink = time + %f;\n          self.attack_finished = time + %f;\n     }\n     else {\n          centerprint(self.owner, \"%s\");\n          self.nextthink = time + 0.1;\n     }\n};\n\n",cur_func, cur_func + 1, work->pause, work->pause + work->show, work->text);
}

char *GetParm (int argc, char **argv, char *s)
{
        int x;
        x = 0;
        while (x < argc)
        {
                if (!strcmp(argv[x++],s))
                        return argv[x];
        }
        return (char *)NULL;
}
                

main(int argc, char **argv)
{
        char *p, configname[256], outputname[256];
        int x;

        start = end = ctrls = 0;
        linenum = 0;
        show = 3;
        delay = 0.1;

        cout << "IntroMaker v1.0 by NiKoDeMoS\n";
        cout << "midctf@quakemania.com\n";
        cout << "http://www.quakemania.com/midctf/\n\n";

        strcpy (configname, "intro.cfg");
        p = GetParm(argc, argv, "-config");
        if (p) strcpy(configname, p); 
        strcpy (outputname, "intro.qc");
        p = GetParm(argc, argv, "-outfile");
        if (p) strcpy(outputname, p);
        strcpy (messagetxt, "Intro by IntroMaker");

        ifstream config(configname);

        if (!config) {
                cout << "Could not open ";
                cout << configname;
                cout << " for input.\n";
                exit(1);
        }

        ofstream outfile(outputname);

        if (!outfile) {
                cout << "Could not open ";
                cout << outputname;
                cout << " for output.\n";
                exit(1);
        }

        while (config && !end) {
                p = buf;
                config.get(*p);
                while (config && *p++ != '\n')
                        config.get(*p);
                *p = (char)NULL;
                linenum++;
                p = buf;
                if (*p != '$') 
                        continue;  // ignore non-controls
                ProcessBuffer ();
        }

        if (!end)
                Error(1, "EOF without end");

        cout << "Processed " << cur_func << " functions with " << ctrls;
        cout << " controls.\n";
        cout << "Generating " << outputname << "...";

        while (cur_func > 0)
        {
                MakeOutputBuffer ();
                outfile.write(buf, strlen(buf));
        }

        cout << "\nDone.\n";

        config.close();
        outfile.close();
}
                       
                

