/*==========================================================================
//  x_rocket.c -- by Patrick Martin             Last updated:  4-14-1999
//--------------------------------------------------------------------------
//  This file contains code that dictates the properties of the
//  rockets (normal and MIRV) in the Armorback patch.
//========================================================================*/

#include "g_local.h"
#include "x_guns.h"
#include "x_rocket.h"


/*=============================/  Bazooka  /=============================*/
/*
   DESCRIPTION:  This is a rocket launcher that shoots rockets with
   more knockback to the attacker so that bigger suits can rocket jump.
*/
/*=======================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This is a near-identical copy of 'T_RadiusDamage'.  This
//  modifies knockback so that heavier suits can rocket-jump.
//------------------------------------------------------------------------*/
void Coven_RadiusDamage (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;
        int     force;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
                points = damage - 0.5 * VectorLength (v);
                if (ent == attacker)
                {       points = points * 0.5;
                        if ((points > 0) && (ent->mass > 200))
                                force = (int)(points * ent->mass / 200);
                        else
                                force = (int)points;
                }
                else
                        force = (int)points;

		if (points > 0)
		{       if (CanDamage (ent, inflictor))
			{       VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
                                T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, force, DAMAGE_RADIUS, mod);
			}
		}
	}
}

/*----------------------------------------------------/ New Code /--------//
//  This is an near-identical copy of 'rocket_touch' except for
//  extra knockback stuff.
//------------------------------------------------------------------------*/
void Coven_RocketTouch
(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
        vec3_t  origin;

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{       G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
		T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_ROCKET);
	else
	{       // don't throw any debris in net games
		if (!deathmatch->value && !coop->value)
		{
			if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        // The only real change.
        Coven_RadiusDamage (ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, MOD_R_SPLASH);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
	gi.multicast (ent->s.origin, MULTICAST_PHS);

	G_FreeEdict (ent);
}

/*----------------------------------------------------/ New Code /--------//
//  This is an identical copy of 'fire_rocket' save for one exception.
//------------------------------------------------------------------------*/
void Coven_FireRocket (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/objects/rocket/tris.md2");
        rocket->owner           = self;
        rocket->touch           = Coven_RocketTouch;
        rocket->nextthink       = level.time + 8000/speed;
        rocket->think           = G_FreeEdict;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");
        rocket->classname       = "rocket";

	if (self->client)
                Coven_CheckDodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*===============================/  MIRV  /===============================*/
/*
   DESCRIPTION:  This fires an rocket that, upon impact, splits
   into several rockets that homes in on a target.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  Do this when the secondary rocket impact or self-destructs.
//------------------------------------------------------------------------*/
void Coven_HomerTouch
(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

        if (ent->master->client)
                PlayerNoise(ent->master, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

	if (other->takedamage)
	{
                T_Damage (other, ent, ent->master, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_DRUNK_ROCKET);
	}

        T_RadiusDamage(ent, ent->master, ent->radius_dmg, other, ent->dmg_radius, MOD_DRUNK_SPLASH);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
        gi.multicast (ent->s.origin, MULTICAST_PHS);

	G_FreeEdict (ent);
}

/*----------------------------------------------------/ New Code /--------//
//  The rocket flies erratically.
//------------------------------------------------------------------------*/
void Coven_DrunkThink (edict_t *self)
{
        vec3_t  forward;

/* If time is up, stop tracking and speed up. */
        if (self->timestamp <= level.time)
        {       self->count *= 2;
                if (self->count > 2000)
                        self->count = 2000;
                VectorScale (self->movedir, self->count, self->velocity);
                self->nextthink = level.time + 8000/self->count;
                self->think = G_FreeEdict;
		return;
	}

/* Fly drunk. */
        self->s.angles[PITCH] += (rand() % 61 - 30);
        self->s.angles[YAW] += (rand() % 61 - 30);
        self->s.angles[ROLL] += (rand() % 61 - 30);
        AngleVectors (self->s.angles, forward, NULL, NULL);
        VectorCopy (forward, self->movedir);
        VectorScale (forward, self->count, self->velocity);

        self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  The secondary rocket tracks its target.  If the rocket has no
//  target or loses it, it employs drunk flight.
//------------------------------------------------------------------------*/
void Coven_HomerThink (edict_t *self)
{
        vec3_t  olddir, newdir;
        vec3_t  spot;

/* If time is up, stop tracking and speed up. */
        if (self->timestamp <= level.time)
        {
//
                vec3_t  ideal;
                vec3_t  forward;
                float   dot;

                VectorSubtract (self->enemy->s.origin, self->s.origin, ideal);
                VectorNormalize (ideal);
                dot = DotProduct (ideal, self->movedir);

                // cos(30) = 0.5. -- cos(60) = ~0.866. //
                if (dot >= 0.5)
                {       // Point straight at the target. //
                        vectoangles (ideal, self->s.angles);
                        VectorCopy (ideal, self->movedir);
                }
                else if (dot != -1)
                {       // Point toward but not directly at target. //
                        AngleVectors (self->s.angles, forward, NULL, NULL);
                        if (dot >= 0)
                                VectorMA (forward, 2, ideal, ideal);
                        else
                                VectorAdd (forward, ideal, ideal);
                        VectorNormalize (ideal);
                        vectoangles (ideal, self->s.angles);
                        VectorCopy (ideal, self->movedir);
                }
//
                self->count = ceil(self->count * 2);
                if (self->count > 2000)
                        self->count = 2000;
                VectorScale (self->movedir, self->count, self->velocity);
                self->nextthink = level.time + 8000/self->count;
                self->think = G_FreeEdict;
		return;
	}

/* If missile loses its target, begin drunk flight. */
        if ((!self->enemy) || (self->enemy->health <= self->enemy->gib_health))
        {       self->think     = Coven_DrunkThink;
                Coven_DrunkThink (self);
                return;
        }

/* Home in on target. */
        VectorAdd (self->enemy->mins, self->enemy->maxs, spot);
        VectorMA (self->enemy->s.origin, 0.5, spot, spot);

        VectorCopy (self->movedir, olddir);
        VectorSubtract (spot, self->s.origin, newdir);
        VectorNormalize (newdir);
        VectorMA (olddir, random() * 2, newdir, newdir);
        VectorNormalize (newdir);
        vectoangles (newdir, self->s.angles);
//
        self->s.angles[PITCH] += (rand() % 61 - 30);
        self->s.angles[YAW] += (rand() % 61 - 30);
        self->s.angles[ROLL] += (rand() % 61 - 30);
        AngleVectors (self->s.angles, newdir, NULL, NULL);
//
        VectorCopy (newdir, self->movedir);
        VectorScale (newdir, self->count, self->velocity);

        self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This sets the initial direction of the secondary rocket.
//------------------------------------------------------------------------*/
void Coven_StartDir (vec3_t dir, vec3_t spread)
{
        vec3_t  v;

        vectoangles (dir, v);
        v[PITCH] += (random() - 0.5) * spread[PITCH];
        v[YAW]   += (random() - 0.5) * spread[YAW];
        v[ROLL]  += (random() - 0.5) * spread[ROLL];

        AngleVectors (v, dir, NULL, NULL);
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns a secondary rocket from the MIRV.
//------------------------------------------------------------------------*/
void Coven_SpawnHomer
(edict_t *self, edict_t *targ, int speed, int damage, float damage_radius, int radius_damage, float seconds)
{
	edict_t	*rocket;
        vec3_t  dir;
        vec3_t  spread = {60, 60, 0};

/* Acquire initial direction. */
        VectorCopy (self->pos1, dir);
        Coven_StartDir (dir, spread);

/* Spawn the rocket. */
	rocket = G_Spawn();
        VectorCopy (self->s.origin, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/cruise/tris.md2");
        rocket->s.frame         = 1;
        rocket->master          = self->owner;
        rocket->enemy           = targ;
        rocket->touch           = Coven_HomerTouch;
        rocket->count           = speed;
        rocket->timestamp       = level.time + seconds;
        rocket->nextthink       = level.time + FRAMETIME;
        if (targ)
                rocket->think   = Coven_HomerThink;
        else
                rocket->think   = Coven_DrunkThink;
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");
	gi.linkentity (rocket);
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns a blank entities to keep track of MIRV targets.
//------------------------------------------------------------------------*/
void Coven_AddTarget (edict_t *self, edict_t *targ)
{
        edict_t *node;

        node = G_Spawn();
        node->flags             |= FL_TEAMSLAVE;
        node->teamchain         = self->teamchain;
        node->teammaster        = self;
        self->teamchain         = node;
        node->owner             = self->owner;
        node->enemy             = targ;
        node->classname         = "node";
        node->movetype          = MOVETYPE_NONE;
        node->solid             = SOLID_NOT;
        node->s.modelindex      = 1;    // must be non-zero
        VectorClear (node->mins);
        VectorClear (node->maxs);
        gi.linkentity (node);
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns more rockets from the MIRV.
//------------------------------------------------------------------------*/
void Coven_MIRV_Split (edict_t *rocket)
{
        edict_t *ent = NULL;
	edict_t	*next;
        int     damage;
        int     radius_damage;
        float   damage_radius;

/* Abort if no rockets are in the MIRV. */
        if (rocket->count < 1)
                return;

/* Set warhead damage. */
        radius_damage = (int)rocket->pos2[2];
        damage_radius = (float)(radius_damage + 40);

/* Push initial target to the end of list, if it is not destroyed. */
        if (rocket->enemy)
                if (rocket->enemy->health > rocket->enemy->gib_health)
                        Coven_AddTarget (rocket, rocket->enemy);

/* Search for more entities to pop on the list. */
        while ((ent = findradius(ent, rocket->s.origin, 1000)) != NULL)
        {
                /* Target must be visible to rocket. */
                if (!CanDamage (ent, rocket))
                        continue;

                /* Target must be visible to attacker. */
                if (!CanDamage (ent, rocket->owner))
                        continue;

                /* Don't bother with impervious entities. */
                if (!ent->takedamage)
                        continue;

                /* Don't track the owner or his allies. */
                if ((ent == rocket->owner) || (Coven_OnSameSide (ent, rocket->owner)))
                        continue;

                /* Initial target already acquired. */
                if (ent == rocket->enemy)
                        continue;

                Coven_AddTarget (rocket, ent);
        }

/* Cycle through the list of entities. */
        if (rocket->teamchain)
        {
                ent = rocket->teamchain;
                while (rocket->count > 0)
                {
                        damage = (int)(rocket->pos2[0] + (random() * rocket->pos2[1]));
                        Coven_SpawnHomer (rocket, ent->enemy, (rand() % 101 + 400), damage, damage_radius, radius_damage, (random() * 0.5 + 5));

                        /* Go to next entity, or recycle if the last. */
                        if (ent->teamchain)
                                ent = ent->teamchain;
                        else
                                ent = rocket->teamchain;

                        (rocket->count)--;
                }
        }
        else
        {       /* No entities in list. */
                while (rocket->count > 0)
                {
                        damage = (int)(rocket->pos2[0] + (random() * rocket->pos2[1]));
                        Coven_SpawnHomer (rocket, NULL, (rand() % 101 + 400), damage, damage_radius, radius_damage, (random() * 0.5 + 5));
                        (rocket->count)--;
                }
        }

/* Pop the entity list by deleting all the blank entities. */
        for (ent = rocket->teamchain; ent; ent = next)
	{       next = ent->teamchain;
                G_FreeEdict (ent);
	}
}

/*----------------------------------------------------/ New Code /--------//
//  Do this when a MIRV touches a target.
//------------------------------------------------------------------------*/
void Coven_MIRV_Touch (edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t		origin;

	if (other == ent->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (ent);
		return;
	}

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

/*
        if (plane)
                VectorCopy (plane->normal, ent->pos1);
*/

        if (other->takedamage)
	{
                if (!Coven_OnSameSide (other, ent->owner))
                        ent->enemy = other;
                T_Damage (other, ent, ent->owner, ent->velocity, ent->s.origin, plane->normal, ent->dmg, 0, 0, MOD_MIRV_ROCKET);
	}
        else
	{
		// don't throw any debris in net games
		if (!deathmatch->value)
		{
                        if ((surf) && !(surf->flags & (SURF_WARP|SURF_TRANS33|SURF_TRANS66|SURF_FLOWING)))
			{
                                int     n;

				n = rand() % 5;
				while(n--)
					ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
			}
		}
	}

        Coven_RadiusDamage (ent, ent->owner, ent->radius_dmg, other, ent->dmg_radius, MOD_MIRV_SPLASH);
        Coven_MIRV_Split (ent);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
        gi.multicast (ent->s.origin, MULTICAST_PHS);

	G_FreeEdict (ent);
}

/*----------------------------------------------------/ New Code /--------//
//  Do this when a MIRV times out.
//------------------------------------------------------------------------*/
void Coven_MIRV_Explode (edict_t *ent)
{
	vec3_t		origin;

	if (ent->owner->client)
		PlayerNoise(ent->owner, ent->s.origin, PNOISE_IMPACT);

	// calculate position for the explosion entity
	VectorMA (ent->s.origin, -0.02, ent->velocity, origin);

        // don't throw any debris in net games
        if (!deathmatch->value)
        {
                 int     n;

                 n = rand() % 5;
                 while(n--)
                        ThrowDebris (ent, "models/objects/debris2/tris.md2", 2, ent->s.origin);
	}

        Coven_RadiusDamage (ent, ent->owner, ent->radius_dmg, NULL, ent->dmg_radius, MOD_MIRV_SPLASH);
        Coven_MIRV_Split (ent);

	gi.WriteByte (svc_temp_entity);
	if (ent->waterlevel)
		gi.WriteByte (TE_ROCKET_EXPLOSION_WATER);
	else
		gi.WriteByte (TE_ROCKET_EXPLOSION);
	gi.WritePosition (origin);
        gi.multicast (ent->s.origin, MULTICAST_PHS);

	G_FreeEdict (ent);
}

/*----------------------------------------------------/ New Code /--------//
//  The MIRV follows its target.
//------------------------------------------------------------------------*/
void Coven_MIRV_Home (edict_t *self)
{
        vec3_t  olddir, newdir;
        vec3_t  spot;

/* If time is up, stop tracking and explode. */
        if (self->timestamp <= level.time)
        {       Coven_MIRV_Explode (self);
                return;
	}

/* If missile loses its target, resume dumb flight at current velocity. */
        if ((!self->enemy) || (self->enemy->health <= self->enemy->gib_health))
        {       self->nextthink = self->timestamp;
                self->think     = Coven_MIRV_Explode;
                return;
        }

/* Home in on target. */
        VectorAdd (self->enemy->mins, self->enemy->maxs, spot);
        VectorMA (self->enemy->s.origin, 0.5, spot, spot);

        VectorCopy (self->movedir, olddir);
        VectorSubtract (spot, self->s.origin, newdir);
        VectorNormalize (newdir);
        VectorMA (olddir, random() * 2, newdir, newdir);
        VectorNormalize (newdir);
        vectoangles (newdir, self->s.angles);
        VectorCopy (newdir, self->movedir);
        VectorScale (newdir, self->style, self->velocity);

        self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This acquires a target that is directly in front of the attacker.
//------------------------------------------------------------------------*/
edict_t *Coven_MIRV_Target (edict_t *ent, vec3_t start, vec3_t dir)
{
        trace_t tr;
        vec3_t  end;

	VectorMA (start, 8192, dir, end);
        tr = gi.trace (start, NULL, NULL, end, ent, MASK_SHOT);
        if (tr.fraction < 1)
                if (tr.ent->takedamage)
                        if (tr.ent->solid == SOLID_BBOX)
                                if (!Coven_OnSameSide (tr.ent, ent))
                                        return tr.ent;

        return NULL;
}

/*----------------------------------------------------/ New Code /--------//
//  Entity launches a MIRV.
//------------------------------------------------------------------------*/
void Coven_FireMIRV (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage, int warheads, vec3_t homer_damage, edict_t *targ)
{
	edict_t	*rocket;

	rocket = G_Spawn();
	VectorCopy (start, rocket->s.origin);
	VectorCopy (dir, rocket->movedir);
	vectoangles (dir, rocket->s.angles);
	VectorScale (dir, speed, rocket->velocity);
        VectorCopy (dir, rocket->pos1);
        VectorInverse (rocket->pos1);
        rocket->movetype        = MOVETYPE_FLYMISSILE;
        rocket->clipmask        = MASK_SHOT;
        rocket->solid           = SOLID_BBOX;
        rocket->s.effects       |= EF_ROCKET;
	VectorClear (rocket->mins);
	VectorClear (rocket->maxs);
        rocket->s.modelindex    = gi.modelindex ("models/cruise/tris.md2");
        rocket->s.frame         = 0;
        rocket->owner           = self;
        rocket->touch           = Coven_MIRV_Touch;
        rocket->timestamp       = level.time + 8000/speed;
        rocket->nextthink       = rocket->timestamp;
        rocket->think           = Coven_MIRV_Explode;
        VectorCopy (homer_damage, rocket->pos2);
        rocket->dmg             = damage;
        rocket->radius_dmg      = radius_damage;
        rocket->dmg_radius      = damage_radius;
        rocket->count           = warheads;
        rocket->teammaster      = rocket;
        rocket->teamchain       = NULL;
        rocket->s.sound         = gi.soundindex ("weapons/rockfly.wav");
        if (targ)
        {       /* We have a lock on target, chase it! */
                rocket->enemy           = targ;
                rocket->style           = speed;
                rocket->nextthink       = level.time + FRAMETIME;
                rocket->think           = Coven_MIRV_Home;
        }

/* Yeah, let the Stroggs die trying to dodge a cruise missile... :) */
	if (self->client)
                Coven_CheckDodge (self, rocket->s.origin, dir, speed);

	gi.linkentity (rocket);
}


/*===========================/  END OF FILE  /===========================*/
