/*==========================================================================
//  x_guns.c -- by Patrick Martin               Last updated:  3-11-1999
//--------------------------------------------------------------------------
//  This file contains code that dictates the properties of the
//  various guns in the Armorback patch.
//
//  NOTE:  Code for some weapons are in their own modules.
//========================================================================*/

#include "g_local.h"
#include "x_guns.h"

extern void Coven_SfxBump (edict_t *ent);


/*=========================/  Generic Code  /=========================*/

/*-------------------------------------------------------- Mirror Code -----
//  This is a copy of 'P_ProjectSource' found in 'p_weapons.c'.
//  Because that function is static, and I do not want to remove
//  the static keyword, a copy is placed here.
//------------------------------------------------------------------------*/
void Coven_ProjectGunSource
(gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
	vec3_t	_distance;

	VectorCopy (distance, _distance);
	if (client->pers.hand == LEFT_HANDED)
		_distance[1] *= -1;
	else if (client->pers.hand == CENTER_HANDED)
		_distance[1] = 0;
	G_ProjectSource (point, _distance, forward, right, result);
}

/*-------------------------------------------------------- Mirror Code -----
//  This is a copy of 'check_dodge' found in 'g_weapons.c'.
//  Because that function is static, and I do not want to
//  modify that file, a copy is placed here.
//------------------------------------------------------------------------*/
void Coven_CheckDodge (edict_t *self, vec3_t start, vec3_t dir, int speed)
{
	vec3_t	end;
	vec3_t	v;
	trace_t	tr;
	float	eta;

	/* easy mode only ducks one quarter the time */
	if (skill->value == 0)
	{
		if (random() > 0.25)
			return;
	}
	VectorMA (start, 8192, dir, end);
	tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);
	if ((tr.ent) && (tr.ent->svflags & SVF_MONSTER) && (tr.ent->health > 0) && (tr.ent->monsterinfo.dodge) && infront(tr.ent, self))
	{
		VectorSubtract (tr.endpos, start, v);
		eta = (VectorLength(v) - tr.ent->maxs[0]) / speed;
		tr.ent->monsterinfo.dodge (tr.ent, self, eta);
	}
}

/*-------------------------------------------------------- Mirror Code -----
//  This tosses a creature through the air against its will.
//------------------------------------------------------------------------*/
void Coven_ThrowThing (edict_t *ent, vec3_t dir, int knockback)
{
        float   mass;

/* Don't throw parts of the map (such as shootable buttons/walls) away. */
        if (ent->solid == SOLID_BSP)
                return;

/* Don't throw crucified soldiers or turret drivers. */
        if (!deathmatch->value)
        {
                if (!strcmp(ent->classname, "misc_insane"))
                        if (ent->spawnflags & 8)
                                if (ent->health > ent->gib_health)
                                        return;

                if (!strcmp(ent->classname, "turret_driver"))
                        return;
        }

/* Insure some mimimum mass. */
	if (ent->mass < 50)
		mass = 50;
	else
		mass = ent->mass;

/* Calculate entity's new velocity. */
	VectorNormalize (dir);
	VectorScale (dir, (knockback / mass), ent->velocity);

/* If entity is on the ground, lift it off the ground. */
	if (ent->groundentity)
	{       if (ent->velocity[2] < 200)
			ent->velocity[2] = 200;
		ent->groundentity = NULL;
	}
}


/*===========================/  Push Command  /===========================*/
/*
   DESCRIPTION:  This pushes targets away from attacker unharmed.
   This can also trigger secret doors, buttons, and the like.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This makes the attacker push things around.
//------------------------------------------------------------------------*/
void Coven_PushIt
(edict_t *self, vec3_t start, vec3_t aimdir, int range)
{
        vec3_t  end;
        trace_t tr;

        VectorMA (start, range, aimdir, end);
        tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);

/* If sky is touched, assume a miss. */
        if ((tr.surface) && (tr.surface->flags & SURF_SKY))
                return;

/* Shove! */
        if (tr.fraction < 1)
        {
                if (tr.ent->takedamage)
                {
                        // Heavier suits push harder.
                        int     force = self->mass * 250;

                        if (tr.ent->solid == SOLID_BSP)
                        {       /* If solid, trigger it if possible. */
                                if (tr.ent->health <= 10)
                                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, tr.ent->health, tr.ent->health, 0, MOD_UNKNOWN);
                                return;
                        }

                        /* Push target away. */
                        if ((tr.ent->client) && (!tr.ent->deadflag))
                                Coven_SfxBump (tr.ent);
                        Coven_ThrowThing (tr.ent, aimdir, force);
                }
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This makes the player shove.
//------------------------------------------------------------------------*/
void Coven_PlayerPush (edict_t *self, int range)
{
	vec3_t	forward, right;
        vec3_t  start;
	vec3_t	offset;

/* Ignore if ghost or dead. */
        if (self->client->resp.spectator || self->deadflag)
                return;

/* Player uses off-hand to shove. */
        AngleVectors (self->client->v_angle, forward, right, NULL);
        VectorSet (offset, 0, -8, self->viewheight-8);
        VectorAdd (offset, vec3_origin, offset);
        Coven_ProjectGunSource (self->client, self->s.origin, offset, forward, right, start);

        Coven_PushIt (self, start, forward, range);
}


/*==========================/  Particle Beam  /==========================*/
/*
   DESCRIPTION:  This shoots an invisible antimatter beam that
   generates small explosions on contact.
*/
/*========================================================================*/

/*------------------------------------------------------/ New Code /--------
//  This animates the small explosion.  The magic numbers in this
//  function represent the frame numbers in id's r_explode md2.
//------------------------------------------------------------------------*/
void Coven_AntiExplodeThink (edict_t *self)
{
        if (++self->s.frame > 4)
	{	G_FreeEdict (self);
		return;
	}

        self->s.skinnum += 2;

        if (self->s.frame == 4)
                self->s.renderfx |= RF_TRANSLUCENT;
        else if (self->s.frame == 3)
		self->s.effects &= ~EF_HYPERBLASTER;

	self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This inflicts constant damage to those within the antimatter
//  explosion.
//------------------------------------------------------------------------*/
void Coven_AntiRadius
(edict_t *inflictor, edict_t *attacker, int die, int base, edict_t *ignore, float radius)
{
        float   points;
	edict_t	*ent = NULL;
	vec3_t	dir;
        vec3_t  spot;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;
                if (!CanDamage (ent, inflictor))
                        continue;

                if (die > 0)
                        points = (rand() % die) + base + 1;
                else
                        points = base;

                if (ent == attacker)
                        points *= 0.5;

                VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
                VectorNormalize2 (dir, spot);
                VectorMA (ent->s.origin, -4, spot, spot);
                T_Damage (ent, inflictor, attacker, dir, spot, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS|DAMAGE_ENERGY, MOD_BEAM_SPLASH);
	}
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns an entity that will become a small, damaging explosion.
//------------------------------------------------------------------------*/
void Coven_SpawnAntimatter
(edict_t *attacker, edict_t *ignore, vec3_t spot, int die, int base, float radius)
{
        edict_t *anti;

        anti = G_Spawn();
        VectorCopy (spot, anti->s.origin);
        VectorClear (anti->mins);
        VectorClear (anti->maxs);
        anti->s.modelindex  = gi.modelindex("models/objects/r_explode/tris.md2");
        anti->s.frame       = 1;
        anti->s.skinnum     = 0;
        anti->touch         = NULL;
        anti->solid         = SOLID_NOT;
        anti->takedamage    = DAMAGE_NO;
        anti->clipmask      = 0;
        anti->s.effects     = EF_HYPERBLASTER;
        anti->s.renderfx    = RF_FULLBRIGHT;
        anti->movetype      = MOVETYPE_NONE;
        VectorClear (anti->velocity);
        VectorClear (anti->s.angles);
        anti->s.angles[YAW] = rand() % 360;
        anti->owner         = NULL;
        anti->enemy         = NULL;
        anti->classname     = "antimatter";
        anti->nextthink     = level.time + FRAMETIME;
        anti->think         = Coven_AntiExplodeThink;
        gi.linkentity (anti);

        Coven_AntiRadius (anti, attacker, die, base, ignore, radius);

        if (gi.pointcontents(anti->s.origin) & MASK_WATER)
                gi.sound(anti, CHAN_AUTO, gi.soundindex("weapons/xpld_wat.wav"), 0.5, ATTN_NORM, 0);
        else
                gi.sound(anti, CHAN_AUTO, gi.soundindex("weapons/grenlx1a.wav"), 0.5, ATTN_NORM, 0);
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns an invisible beam.
//
//  NOTE:  'damage' vector fields --
//         [0] = Direct hit damage.
//         [1] = Random splash damage.
//         [2] = Base splash damage.
//------------------------------------------------------------------------*/
void Coven_FireBeam
(edict_t *self, vec3_t start, vec3_t aimdir, vec3_t spread, vec3_t damage, int range)
{
        vec3_t  end;
        trace_t tr;
        vec3_t  v;

/* Change aiming direction. */
        vectoangles (aimdir, v);
        v[PITCH] += (random() - 0.5) * spread[PITCH];
        v[YAW]   += (random() - 0.5) * spread[YAW];
        v[ROLL]  += (random() - 0.5) * spread[ROLL];
        AngleVectors (v, aimdir, NULL, NULL);
        VectorNormalize (aimdir);

        VectorMA (start, range, aimdir, end);
        tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);

/* Burn the target and other targets near it. */
        if ((tr.fraction < 1) && (tr.surface) && (!(tr.surface->flags & SURF_SKY)))
        {
                if (tr.ent->takedamage)
                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, (int)damage[0], (int)damage[0], DAMAGE_ENERGY, MOD_BEAM);

                Coven_SpawnAntimatter (self, tr.ent, tr.endpos, (int)damage[1], (int)damage[2], 80);
        }

	if (self->client)
                PlayerNoise (self, tr.endpos, PNOISE_IMPACT);
}


/*=============================/  Prod Rod  /=============================*/
/*
   DESCRIPTION:  This damages and pushes away one target close and
   in front of the attacker.  It can be used as a hi-tech pogo stick.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This is called the prod blaster bolt touches something.
//------------------------------------------------------------------------*/
void Coven_ProdBoltTouch
(edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	if (other == self->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{       G_FreeEdict (self);
		return;
	}

	if (self->owner->client)
                PlayerNoise (self->owner, self->s.origin, PNOISE_IMPACT);

	if (other->takedamage)
	{
                T_Damage (other, self, self->owner, self->velocity, self->s.origin, plane->normal, self->dmg, 1, DAMAGE_ENERGY, MOD_BLASTER);
	}
	else
	{
		gi.WriteByte (svc_temp_entity);
		gi.WriteByte (TE_BLASTER);
		gi.WritePosition (self->s.origin);
		if (!plane)
			gi.WriteDir (vec3_origin);
		else
			gi.WriteDir (plane->normal);
		gi.multicast (self->s.origin, MULTICAST_PVS);
	}

	G_FreeEdict (self);
}

/*----------------------------------------------------/ New Code /--------//
//  This fires the poker.  This is similar to the Quake1 axe.
//------------------------------------------------------------------------*/
void Coven_FirePoker
(edict_t *self, vec3_t start, vec3_t aimdir, int range, int damage, int blast_damage)
{
        vec3_t  dir;
        vec3_t  end;
        trace_t tr;
        int     maxforce = self->mass * 500;

        VectorMA (start, range, aimdir, end);

        tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);

/* If sky is touched, assume a miss. */
        if ((tr.surface) && (tr.surface->flags & SURF_SKY))
                return;

/* Zap target. */
        if (tr.fraction < 1)
        {
                int     force;
                int     mass = 0;

                gi.sound (self, CHAN_AUTO, gi.soundindex("weapons/poker/pokehit.wav"), 1, ATTN_NORM, 0);

        /* If target can be damaged, hurt it and push it away. */
                if (tr.ent->takedamage)
                {
                        VectorCopy (aimdir, dir);
                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, damage, damage, 0, MOD_POKER);
                        Coven_ThrowThing (tr.ent, dir, 300000);
                        if (tr.ent->mass < 50)
                                mass = 50;
                        else
                                mass = tr.ent->mass;
                }
                else
                {       /* Show some sparks. */
                        gi.WriteByte (svc_temp_entity);
                        gi.WriteByte (TE_BLASTER);
                        gi.WritePosition (tr.endpos);
                        gi.WriteDir (tr.plane.normal);
                        gi.multicast (tr.endpos, MULTICAST_PVS);
                }

        /* Push the attacker as well (though not as hard as the target). */
                force = 100 * mass;
                if ((!force) || (force > maxforce))
                        force = maxforce;
                VectorSubtract (start, tr.endpos, dir);
                Coven_ThrowThing (self, dir, force);
        }
        else
        {
                edict_t *bolt;
                float   speed = 2000;

        /* Create the bolt. */
                bolt = G_Spawn();
                bolt->s.modelindex    = gi.modelindex ("models/objects/laser/tris.md2");
                bolt->s.frame         = 0;
                bolt->s.skinnum       = 0;
                VectorClear (bolt->mins);
                VectorClear (bolt->maxs);
                VectorCopy (start, bolt->s.origin);
                vectoangles (aimdir, bolt->s.angles);
                VectorScale (aimdir, speed, bolt->velocity);
                bolt->movetype        = MOVETYPE_FLYMISSILE;
                bolt->clipmask        = MASK_SHOT;
                bolt->solid           = SOLID_BBOX;
                bolt->takedamage      = DAMAGE_NO;
                bolt->s.effects       = EF_IONRIPPER;
                bolt->s.renderfx      = 0;
                bolt->owner           = self;
                bolt->classname       = "prod bolt";
                bolt->touch           = Coven_ProdBoltTouch;
                bolt->nextthink       = level.time + 600/speed;
                bolt->think           = G_FreeEdict;
                bolt->dmg             = blast_damage;
                bolt->s.sound         = gi.soundindex ("misc/lasfly.wav");
                gi.linkentity (bolt);

        /* Check for monster dodge routine. */
                if (self->client)
                        Coven_CheckDodge (self, bolt->s.origin, aimdir, speed);
        }

	if (self->client)
                PlayerNoise (self, tr.endpos, PNOISE_IMPACT);
}


/*==========================/  Photon Cannon  /==========================*/
/*
   DESCRIPTION:  This fires a beam that will damage and throw the target.
   If a suitable target is struck, more targets near it are also fried.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This searches for entities nearby a target.  Any entity found
//  will be attacked by either splash damage or another photon beam.
//
//  NOTE:  'storm' vector fields --
//         [0] = Chain beam attack radius.
//         [1] = Splash damage radius (*never* greater than beam radius).
//         [2] = Maximum damage possible from beam/splash damage.
//------------------------------------------------------------------------*/
void Coven_PhotonStorm
(edict_t *targ, edict_t *attacker, vec3_t start, vec3_t storm, int limit)
{
	edict_t	*ent = NULL;
        vec3_t  v;
        vec3_t  spot;
        trace_t tr;
        float   points;
        float   dist;

        while ((ent = findradius(ent, targ->s.origin, storm[0])) != NULL)
        {
                if (!CanDamage (ent, targ))
                        continue;

                if (!ent->takedamage)
                        continue;

                if (ent == targ)
                        continue;

                /* Calculate entity's center. */
                VectorAdd (ent->mins, ent->maxs, v);
                VectorMA (ent->s.origin, 0.5, v, v);
                tr = gi.trace (start, NULL, NULL, v, targ, MASK_SHOT);

                if ( (limit >= 1) && (tr.ent == ent) && (ent != attacker) && (!Coven_OnSameSide (ent, attacker)) )
                {       /* Trace beam. */
                        gi.WriteByte (svc_temp_entity);
                        gi.WriteByte (TE_BFG_LASER);
                        gi.WritePosition (start);
                        gi.WritePosition (v);
                        gi.multicast (targ->s.origin, MULTICAST_PHS);

                        /* Apply the damage. */
                        if (tr.ent->takedamage)
                        {       VectorSubtract (tr.ent->s.origin, targ->s.origin, v);
                                T_Damage (tr.ent, targ, attacker, vec3_origin, tr.endpos, vec3_origin, (int)storm[2], (int)storm[2], DAMAGE_ENERGY, MOD_P_CHAIN);
                        }

                        /* One beam used.*/
                        limit--;
                }
                else
                {       /* Normal splash damage. */
                        VectorSubtract (targ->s.origin, v, v);
			dist = VectorLength(v);
                        points = (storm[2] * (1.0 - (dist/storm[1])));

                        if (ent == attacker)
                                points *= 0.5;

                        if (points > 0)
                        {       VectorSubtract (ent->s.origin, targ->s.origin, v);
                                VectorNormalize2 (v, spot);
                                VectorMA (ent->s.origin, -4, spot, spot);
                                T_Damage (ent, targ, attacker, v, spot, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS|DAMAGE_ENERGY, MOD_P_SPLASH);
                        }
                }
        }
}

/*----------------------------------------------------/ New Code /--------//
//  This spawns a blank entity whose purpose is to fry entities
//  near the photon explosion.
//------------------------------------------------------------------------*/
void Coven_SpawnZapper (edict_t *attacker, vec3_t origin, vec3_t storm)
{
        edict_t *zap;

        zap = G_Spawn();
        VectorCopy (origin, zap->s.origin);
        VectorClear (zap->s.angles);
        VectorClear (zap->velocity);
        zap->movetype        = MOVETYPE_NONE;
        zap->solid           = SOLID_NOT;
        zap->clipmask        = 0;
        VectorClear (zap->mins);
        VectorClear (zap->maxs);
        zap->s.modelindex    = gi.modelindex ("");
        zap->touch           = NULL;
        zap->classname       = "zapper";
        gi.linkentity (zap);

        Coven_PhotonStorm (zap, attacker, origin, storm, 0);

        G_FreeEdict (zap);
}

/*----------------------------------------------------/ New Code /--------//
//  This animates then eliminates the photon explosion entity.
//------------------------------------------------------------------------*/
void Coven_PX_Think (edict_t *self)
{
        if (++self->s.frame > 3)
        {       G_FreeEdict (self);
                return;
        }

	self->nextthink = level.time + FRAMETIME;
}

/*----------------------------------------------------/ New Code /--------//
//  This creates a big, bright green explosion.
//------------------------------------------------------------------------*/
void Coven_PhotonExplosion (vec3_t origin)
{
        edict_t *ent;

/* Create a green flash. */
        gi.WriteByte (svc_temp_entity);
        gi.WriteByte (TE_BFG_EXPLOSION);
        gi.WritePosition (origin);
        gi.multicast (origin, MULTICAST_PVS);

/* Create a green particle explosion effect. */
        gi.WriteByte (svc_temp_entity);
        gi.WriteByte (TE_BFG_BIGEXPLOSION);
        gi.WritePosition (origin);
        gi.multicast (origin, MULTICAST_PVS);

/* Spawn an explosion entity for use as noisemaker. */
        ent = G_Spawn();
        VectorCopy (origin, ent->s.origin);
        VectorClear (ent->s.angles);
        VectorClear (ent->velocity);

        ent->s.modelindex  = gi.modelindex ("sprites/s_bfg2.sp2");
        ent->s.frame       = 1;
        ent->s.skinnum     = 0;
        ent->touch         = NULL;
        ent->solid         = SOLID_NOT;
        ent->takedamage    = DAMAGE_NO;
        ent->movetype      = MOVETYPE_NONE;
        ent->clipmask      = 0;
        ent->s.effects     = 0;
        ent->s.renderfx    = 0;
        ent->classname     = "photon";
        ent->nextthink     = level.time + FRAMETIME;
        ent->think         = Coven_PX_Think;

        gi.linkentity (ent);

/* Make some noise. */
        if (gi.pointcontents(origin) & MASK_WATER)
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/xpld_wat.wav"), 1, ATTN_NORM, 0);
        else
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/grenlx1a.wav"), 1, ATTN_NORM, 0);
}

/*----------------------------------------------------/ New Code /--------//
//  Fire photon cannon.  A green beam is drawn straight ahead.
//  If it touches something, a greenish explosion will erupt and
//  singe everyone nearby.  In addition, if the beam strikes a
//  shootable target, more beams will erupt to attack more targets
//  nearby.
//------------------------------------------------------------------------*/
void Coven_FirePhoton
(edict_t *self, vec3_t start, vec3_t center, vec3_t aimdir, vec3_t storm, int damage, int limit)
{
        vec3_t  end;
        trace_t tr;

/* Aim and trace exactly to where the crosshairs point. */
        VectorMA (center, 8192, aimdir, end);
        tr = gi.trace (center, NULL, NULL, end, self, MASK_SHOT);

/* Draw beam & flash.  Beam starts at the gun barrel. */
	gi.WriteByte (svc_temp_entity);
        gi.WriteByte (TE_BFG_LASER);
	gi.WritePosition (start);
	gi.WritePosition (tr.endpos);
	gi.multicast (self->s.origin, MULTICAST_PHS);

        if ((tr.fraction < 1) && (tr.surface) && (!(tr.surface->flags & SURF_SKY)))
        {
                vec3_t  origin;

                /* Calculate position for explosion entity. */
                VectorMA (tr.endpos, -8, aimdir, origin);

                /* Zap the target and search for other targets near it. */
                if (tr.ent->takedamage)
                {
                        vec3_t  dir;

                        Coven_PhotonStorm (tr.ent, self, tr.endpos, storm, limit);
                        VectorCopy (aimdir, dir);
                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, damage, damage, DAMAGE_ENERGY, MOD_PHOTON);
                        Coven_ThrowThing (tr.ent, dir, 300000);
                }
                else
                {
                        Coven_SpawnZapper (self, tr.endpos, storm);
                }

                Coven_PhotonExplosion (origin);
        }

	if (self->client)
                PlayerNoise (self, tr.endpos, PNOISE_IMPACT);
}


/*==============================/  Carver  /==============================*/
/*
   DESCRIPTION:  This damages the target in front of the attacker.
*/
/*========================================================================*/

/*----------------------------------------------------/ New Code /--------//
//  This fires the saw.  This is similar to the Quake1 axe.
//------------------------------------------------------------------------*/
void Coven_FireSaw
(edict_t *self, vec3_t start, vec3_t aimdir, int damage, int range, qboolean kill)
{
        vec3_t  end;
        trace_t tr;

        VectorMA (start, range, aimdir, end);

        tr = gi.trace (start, NULL, NULL, end, self, MASK_SHOT);

/* If sky is touched, assume a miss. */
        if ((tr.surface) && (tr.surface->flags & SURF_SKY))
                return;

/* Slice target. */
        if (tr.fraction < 1)
        {
        /* If target can be damaged, hurt it and pull it in slightly. */
                if (tr.ent->takedamage)
                {       /* Make some more sparks. */
                        if ((tr.ent->svflags & SVF_MONSTER) || (tr.ent->client))
                        {       /* Struck some meat. */
                                gi.sound (self, CHAN_WEAPON, gi.soundindex("weapons/carver/sawhit.wav"), 1, ATTN_NORM, 0);
                                gi.WriteByte (svc_temp_entity);
                                gi.WriteByte (TE_BLOOD);
                                gi.WritePosition (tr.endpos);
                                gi.WriteDir (vec3_origin);
                                gi.multicast (tr.endpos, MULTICAST_PVS);
                        }
                        else
                        {       /* Struck some hard surface. */
                                gi.sound (self, CHAN_WEAPON, gi.soundindex("weapons/carver/sawhit2.wav"), 1, ATTN_NORM, 0);
                                gi.WriteByte (svc_temp_entity);
                                gi.WriteByte (TE_SPARKS);
                                gi.WritePosition (tr.endpos);
                                gi.WriteDir (vec3_origin);
                                gi.multicast (tr.endpos, MULTICAST_PVS);
                        }

                        /* Hurt it! */
                        VectorInverse (aimdir);
                        if (kill && (tr.ent->mass < MASS_HEAVY))
                        {       /* Don't kill outright in half damage mode. */
                                if ((int)(dmflags->value) & DF_HALF_DAMAGE)
                                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, damage * 2, damage, 0, MOD_CARVER);
                                else
                                        T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, 100000, damage, DAMAGE_NO_ARMOR, MOD_CARVER);
                        }
                        else
                                T_Damage (tr.ent, self, self, aimdir, tr.endpos, tr.plane.normal, damage, damage, 0, MOD_CARVER);
                }
                else
                {       /* Struck something impervious to damage. */
                        gi.sound (self, CHAN_WEAPON, gi.soundindex("weapons/carver/sawhit2.wav"), 1, ATTN_NORM, 0);
                        gi.WriteByte (svc_temp_entity);
                        gi.WriteByte (TE_SPARKS);
                        gi.WritePosition (tr.endpos);
                        gi.WriteDir (tr.plane.normal);
                        gi.multicast (tr.endpos, MULTICAST_PVS);
                }
        }

	if (self->client)
                PlayerNoise (self, tr.endpos, PNOISE_IMPACT);
}


/*===========================/  END OF FILE  /===========================*/
