// g_weapon.c

#include "g_local.h"
#include "m_player.h"
// COVEN:  Include new header files.
#include "x_ctf.h"
#include "x_fire.h"
#include "x_guns.h"
#include "x_hook.h"
#include "x_jet.h"
#include "x_rocket.h"
// COVEN


static qboolean	is_quad;
static byte		is_silenced;


void weapon_grenade_fire (edict_t *ent, qboolean held);


static void P_ProjectSource (gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
	vec3_t	_distance;

	VectorCopy (distance, _distance);
	if (client->pers.hand == LEFT_HANDED)
		_distance[1] *= -1;
	else if (client->pers.hand == CENTER_HANDED)
		_distance[1] = 0;
	G_ProjectSource (point, _distance, forward, right, result);
}


/*
===============
PlayerNoise

Each player can have two noise objects associated with it:
a personal noise (jumping, pain, weapon firing), and a weapon
target noise (bullet wall impacts)

Monsters that don't directly see the player can move
to a noise in hopes of seeing the player from there.
===============
*/
void PlayerNoise(edict_t *who, vec3_t where, int type)
{
	edict_t		*noise;

	if (type == PNOISE_WEAPON)
	{
		if (who->client->silencer_shots)
		{
// COVEN:  Melee weapons do not use silencer.
                        if ((strcmp(who->client->pers.weapon->pickup_name, "Grapple")) &&
                            (strcmp(who->client->pers.weapon->pickup_name, "Prod Rod")) &&
                            (strcmp(who->client->pers.weapon->pickup_name, "Carver")))
                        {
                                who->client->silencer_shots--;
                                return;
                        }
// COVEN
		}
	}

	if (deathmatch->value)
		return;

	if (who->flags & FL_NOTARGET)
		return;


	if (!who->mynoise)
	{
		noise = G_Spawn();
		noise->classname = "player_noise";
		VectorSet (noise->mins, -8, -8, -8);
		VectorSet (noise->maxs, 8, 8, 8);
		noise->owner = who;
		noise->svflags = SVF_NOCLIENT;
		who->mynoise = noise;

		noise = G_Spawn();
		noise->classname = "player_noise";
		VectorSet (noise->mins, -8, -8, -8);
		VectorSet (noise->maxs, 8, 8, 8);
		noise->owner = who;
		noise->svflags = SVF_NOCLIENT;
		who->mynoise2 = noise;
	}

	if (type == PNOISE_SELF || type == PNOISE_WEAPON)
	{
		noise = who->mynoise;
		level.sound_entity = noise;
		level.sound_entity_framenum = level.framenum;
	}
	else // type == PNOISE_IMPACT
	{
		noise = who->mynoise2;
		level.sound2_entity = noise;
		level.sound2_entity_framenum = level.framenum;
	}

	VectorCopy (where, noise->s.origin);
	VectorSubtract (where, noise->maxs, noise->absmin);
	VectorAdd (where, noise->maxs, noise->absmax);
	noise->teleport_time = level.time;
	gi.linkentity (noise);
}


qboolean Pickup_Weapon (edict_t *ent, edict_t *other)
{
	int			index;
	gitem_t		*ammo;

	index = ITEM_INDEX(ent->item);

	if ( ( ((int)(dmflags->value) & DF_WEAPONS_STAY) || coop->value) 
		&& other->client->pers.inventory[index])
	{
		if (!(ent->spawnflags & (DROPPED_ITEM | DROPPED_PLAYER_ITEM) ) )
			return false;	// leave the weapon for others to pickup
	}

	other->client->pers.inventory[index]++;

	if (!(ent->spawnflags & DROPPED_ITEM) )
	{
		// give them some ammo with it
		ammo = FindItem (ent->item->ammo);
		if ( (int)dmflags->value & DF_INFINITE_AMMO )
			Add_Ammo (other, ammo, 1000);
		else
			Add_Ammo (other, ammo, ammo->quantity);

		if (! (ent->spawnflags & DROPPED_PLAYER_ITEM) )
		{
			if (deathmatch->value)
			{
				if ((int)(dmflags->value) & DF_WEAPONS_STAY)
					ent->flags |= FL_RESPAWN;
				else
					SetRespawn (ent, 30);
			}
			if (coop->value)
				ent->flags |= FL_RESPAWN;
		}
	}

// COVEN:  No blaster in AB.
	if (other->client->pers.weapon != ent->item &&
		(other->client->pers.inventory[index] == 1) &&
                ( !deathmatch->value ) )
		other->client->newweapon = ent->item;
// COVEN

	return true;
}


//======================================================================//
// COVEN:  Support functions!
// Check if player should switch hands.
static qboolean Coven_SwitchHands (edict_t *ent)
{
        // Don't change hands if not in DM or if dead.
        if (!deathmatch->value)
                return false;
        if (ent->deadflag)
                return false;

        // Always switch if player has only two weapons.
        if (ab_arms != ARMS_ALL)
                return true;

        // Switch if player is changing from melee to gun or vice versa.
        if (!Q_stricmp(ent->client->pers.lastweapon->pickup_name, "Grapple"))
                return true;
        if (!Q_stricmp(ent->client->pers.lastweapon->pickup_name, "Prod Rod"))
                return true;
        if (!Q_stricmp(ent->client->pers.lastweapon->pickup_name, "Carver"))
                return true;
        if (!Q_stricmp(ent->client->pers.weapon->pickup_name, "Grapple"))
                return true;
        if (!Q_stricmp(ent->client->pers.weapon->pickup_name, "Prod Rod"))
                return true;
        if (!Q_stricmp(ent->client->pers.weapon->pickup_name, "Carver"))
                return true;

        // Player is switching weapon 'modes' for the combo gun...
        return false;
}
// COVEN
//======================================================================//


/*
===============
ChangeWeapon

The old weapon has been dropped all the way, so make the new one
current
===============
*/
void ChangeWeapon (edict_t *ent)
{
	int i;

// COVEN:  No hand grenades in ArmourBack.
/*
	if (ent->client->grenade_time)
	{
		ent->client->grenade_time = level.time;
		ent->client->weapon_sound = 0;
		weapon_grenade_fire (ent, false);
		ent->client->grenade_time = 0;
	}
*/
// COVEN

	ent->client->pers.lastweapon = ent->client->pers.weapon;
	ent->client->pers.weapon = ent->client->newweapon;
	ent->client->newweapon = NULL;
	ent->client->machinegun_shots = 0;

	// set visible model
	if (ent->s.modelindex == 255) {
		if (ent->client->pers.weapon)
			i = ((ent->client->pers.weapon->weapmodel & 0xff) << 8);
		else
			i = 0;
		ent->s.skinnum = (ent - g_edicts - 1) | i;
	}

	if (ent->client->pers.weapon && ent->client->pers.weapon->ammo)
		ent->client->ammo_index = ITEM_INDEX(FindItem(ent->client->pers.weapon->ammo));
	else
		ent->client->ammo_index = 0;

	if (!ent->client->pers.weapon)
	{	// dead
		ent->client->ps.gunindex = 0;
		return;
	}

	ent->client->weaponstate = WEAPON_ACTIVATING;
	ent->client->ps.gunframe = 0;
	ent->client->ps.gunindex = gi.modelindex(ent->client->pers.weapon->view_model);

// COVEN:  Switch hands in DM.
        ent->client->resp.lastgun = ent->client->pers.weapon;

        if (Coven_SwitchHands (ent))
        {
                if (ent->client->pers.hand == RIGHT_HANDED)
                {       // Switch to left hand.
                        gi.WriteByte (svc_stufftext);
                        gi.WriteString ("hand 1");
                        gi.unicast (ent, true);
                }
                else //if (ent->client->pers.hand == LEFT_HANDED)
                {       // Switch to right hand.
                        gi.WriteByte (svc_stufftext);
                        gi.WriteString ("hand 0");
                        gi.unicast (ent, true);
                }
        }

        // Battlesuits always carry a gun in each arm, so don't flinch!
        if (ent->battlesuit)
                return;
// COVEN

	ent->client->anim_priority = ANIM_PAIN;
	if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
	{
			ent->s.frame = FRAME_crpain1;
			ent->client->anim_end = FRAME_crpain4;
	}
	else
	{
			ent->s.frame = FRAME_pain301;
			ent->client->anim_end = FRAME_pain304;
			
	}
}

/*
=================
NoAmmoWeaponChange
=================
*/
void NoAmmoWeaponChange (edict_t *ent)
{
// COVEN:  Change to other AB weapon.
        // Rapid-fire class.
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Shells"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Auto Shotgun"))]) )
        {       ent->client->newweapon = FindItem ("Auto Shotgun");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Bullets"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Minigun"))]) )
        {       ent->client->newweapon = FindItem ("Minigun");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Cells"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Particle Beam"))]) )
        {       ent->client->newweapon = FindItem ("Particle Beam");
                return;
        }

        // Power shot class.
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Fuel"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Flamethrower"))]) )
        {       ent->client->newweapon = FindItem ("Flamethrower");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Cells"))] >= (ent->client->pers.weapon->quantity)) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Photon Cannon"))]) )
        {       ent->client->newweapon = FindItem ("Photon Cannon");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Shells"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Assault Shotgun"))]) )
        {       ent->client->newweapon = FindItem ("Assault Shotgun");
                return;
        }

        // Melee class.
        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Carver"))])
        {       ent->client->newweapon = FindItem ("Carver");
                return;
        }
        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Prod Rod"))])
        {       ent->client->newweapon = FindItem ("Prod Rod");
                return;
        }
        if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Grapple"))])
        {       ent->client->newweapon = FindItem ("Grapple");
                return;
        }

        // Explosive (area effect) class.
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Grenades"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Grenade Launcher"))]) )
        {       ent->client->newweapon = FindItem ("Grenade Launcher");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Rockets"))]) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("Bazooka"))]) )
        {       ent->client->newweapon = FindItem ("Bazooka");
                return;
        }
        if ( (ent->client->pers.inventory[ITEM_INDEX(FindItem("Rockets"))] >= 5) &&
             (ent->client->pers.inventory[ITEM_INDEX(FindItem("MIRV Launcher"))]) )
        {       ent->client->newweapon = FindItem ("MIRV Launcher");
                return;
        }
// COVEN
}

/*
=================
Think_Weapon

Called by ClientBeginServerFrame and ClientThink
=================
*/
void Think_Weapon (edict_t *ent)
{
	// if just died, put the weapon away
	if (ent->health < 1)
	{
		ent->client->newweapon = NULL;
		ChangeWeapon (ent);
	}

	// call active weapon think routine
	if (ent->client->pers.weapon && ent->client->pers.weapon->weaponthink)
	{
		is_quad = (ent->client->quad_framenum > level.framenum);
		if (ent->client->silencer_shots)
			is_silenced = MZ_SILENCED;
		else
			is_silenced = 0;
		ent->client->pers.weapon->weaponthink (ent);
	}
}


/*
================
Use_Weapon

Make the weapon ready if there is ammo
================
*/
void Use_Weapon (edict_t *ent, gitem_t *item)
{
	int			ammo_index;
	gitem_t		*ammo_item;

	// see if we're already using it
	if (item == ent->client->pers.weapon)
		return;

	if (item->ammo && !g_select_empty->value && !(item->flags & IT_AMMO))
	{
		ammo_item = FindItem(item->ammo);
		ammo_index = ITEM_INDEX(ammo_item);

		if (!ent->client->pers.inventory[ammo_index])
		{
			gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}

		if (ent->client->pers.inventory[ammo_index] < item->quantity)
		{
			gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}
	}

	// change to this weapon when down
	ent->client->newweapon = item;
}



/*
================
Drop_Weapon
================
*/
void Drop_Weapon (edict_t *ent, gitem_t *item)
{
	int		index;

	if ((int)(dmflags->value) & DF_WEAPONS_STAY)
		return;

	index = ITEM_INDEX(item);
	// see if we're already using it
	if ( ((item == ent->client->pers.weapon) || (item == ent->client->newweapon))&& (ent->client->pers.inventory[index] == 1) )
	{
		gi.cprintf (ent, PRINT_HIGH, "Can't drop current weapon\n");
		return;
	}

	Drop_Item (ent, item);
	ent->client->pers.inventory[index]--;
}


//======================================================================//
// COVEN:  Support functions!
// Select proper attack animation.
static void Coven_AttackAnim (edict_t *ent)
{
        // Don't animate model if player is dead!
        if (ent->deadflag)
                return;

        if (ent->battlesuit)
        {
                float   xyspeed;
                xyspeed = sqrt(ent->velocity[0]*ent->velocity[0] + ent->velocity[1]*ent->velocity[1]);

                if ((ent->client->ps.pmove.pm_flags & PMF_DUCKED) || (ent->armoured))
                {       // Crouched.
                        ent->s.frame = ent->crouch_attack_anim_start-1;
                        ent->client->anim_end = ent->crouch_attack_anim_end;
                        //gi.cprintf (ent, PRINT_HIGH, "ducked - FIRING\n");
                }
                else if (ent->waterlevel && !ent->groundentity && !Jet_Active(ent))
                {       // Swimming and not on the ground.
                        ent->s.frame = ent->swim_attack_anim_start-1;
                        ent->client->anim_end = ent->swim_attack_anim_end;
                        //gi.cprintf (ent, PRINT_HIGH, "swimming - FIRING\n");
                }
                else if (xyspeed)
                {       // Running.
                        ent->s.frame = ent->run_attack_anim_start-1;
                        ent->client->anim_end = ent->run_attack_anim_end;
                        //gi.cprintf (ent, PRINT_HIGH, "running - FIRING\n");
                }
                else
                {       // Standing.
                        ent->s.frame = ent->attack_anim_start-1;
                        ent->client->anim_end = ent->attack_anim_end;
                        //gi.cprintf (ent, PRINT_HIGH, "standing - FIRING\n");
                }
        }
        else
        {       // PPM attack animation.
                if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
                {       ent->s.frame = FRAME_crattak1-1;
                        ent->client->anim_end = FRAME_crattak9;
                }
                else
                {       ent->s.frame = FRAME_attack1-1;
                        ent->client->anim_end = FRAME_attack8;
                }
        }
}

// Play powered-up attack sound.
static void Coven_PowerupSfx (edict_t *ent)
{
        if (CTFApplyStrengthSound (ent))
                return;

        if ((ent->client->quad_framenum > level.framenum) && (ent->client->quadsndtime < level.time))
        {       ent->client->quadsndtime = level.time + 1;
                gi.sound (ent, CHAN_ITEM, gi.soundindex("items/damage3.wav"), 1, ATTN_NORM, 0);
                return;
        }

        CTFApplyHasteSound (ent);
}

// Return vector to where the crosshairs point exactly.
// NOTE:  Only projectiles should use this.
static void Coven_CrosshairVector
(vec3_t forward, vec3_t start, edict_t *ent)
{
        trace_t tr;
        vec3_t  v;

        VectorCopy (ent->s.origin, v);
        v[2] += ent->viewheight;
        VectorMA (v, 8192, forward, forward);
        tr = gi.trace (v, NULL, NULL, forward, ent, MASK_SHOT);
        VectorSubtract (tr.endpos, start, forward);
        VectorNormalize (forward);
}

// COVEN
//======================================================================//


/*
================
Weapon_Generic

A generic function to handle the basics of weapon thinking
================
*/
#define FRAME_FIRE_FIRST		(FRAME_ACTIVATE_LAST + 1)
#define FRAME_IDLE_FIRST		(FRAME_FIRE_LAST + 1)
#define FRAME_DEACTIVATE_FIRST	(FRAME_IDLE_LAST + 1)

// COVEN:  Borrowed from Zoid's CTF.  Functions heavily modified.
static void Weapon_Generic2 (edict_t *ent, int FRAME_ACTIVATE_LAST, int FRAME_FIRE_LAST, int FRAME_IDLE_LAST, int FRAME_DEACTIVATE_LAST, int *pause_frames, int *fire_frames, void (*fire)(edict_t *ent))
{
        int     n;

	if (ent->client->weaponstate == WEAPON_DROPPING)
	{
		if (ent->client->ps.gunframe == FRAME_DEACTIVATE_LAST)
		{
			ChangeWeapon (ent);
			return;
		}
                else if ( !ent->battlesuit &&
                          ((FRAME_DEACTIVATE_LAST - ent->client->ps.gunframe) == 4) )
		{
			ent->client->anim_priority = ANIM_REVERSE;
			if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
			{
				ent->s.frame = FRAME_crpain4+1;
				ent->client->anim_end = FRAME_crpain1;
			}
			else
			{
				ent->s.frame = FRAME_pain304+1;
				ent->client->anim_end = FRAME_pain301;
				
			}
		}

		ent->client->ps.gunframe++;
		return;
	}

	if (ent->client->weaponstate == WEAPON_ACTIVATING)
	{
                if ( (ent->client->ps.gunframe == FRAME_ACTIVATE_LAST) ||
                     ((int)(dmflags->value) & DF_FAST_SWITCH) )
		{
			ent->client->weaponstate = WEAPON_READY;
			ent->client->ps.gunframe = FRAME_IDLE_FIRST;
			return;
		}

		ent->client->ps.gunframe++;
		return;
	}

	if ((ent->client->newweapon) && (ent->client->weaponstate != WEAPON_FIRING))
	{
                // Check for instant switch.  Pain/switch anim is bypassed as well.
                if ((int)(dmflags->value) & DF_FAST_SWITCH)
		{       // Bypass pain/switch anim as well.
                        ChangeWeapon (ent);
			return;
		}

		ent->client->weaponstate = WEAPON_DROPPING;
		ent->client->ps.gunframe = FRAME_DEACTIVATE_FIRST;

                // Battlesuits don't flinch!
                if ( !ent->battlesuit &&
                     ((FRAME_DEACTIVATE_LAST - FRAME_DEACTIVATE_FIRST) < 4) )
		{
			ent->client->anim_priority = ANIM_REVERSE;
			if(ent->client->ps.pmove.pm_flags & PMF_DUCKED)
			{
				ent->s.frame = FRAME_crpain4+1;
				ent->client->anim_end = FRAME_crpain1;
			}
			else
			{
				ent->s.frame = FRAME_pain304+1;
				ent->client->anim_end = FRAME_pain301;
				
			}
		}
		return;
	}

	if (ent->client->weaponstate == WEAPON_READY)
	{
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{
			ent->client->latched_buttons &= ~BUTTON_ATTACK;
			if ((!ent->client->ammo_index) || 
				( ent->client->pers.inventory[ent->client->ammo_index] >= ent->client->pers.weapon->quantity))
			{
                                // Check if players are allowed to shoot.
                                if (Coven_NoFire ())
                                        return;

				ent->client->ps.gunframe = FRAME_FIRE_FIRST;
				ent->client->weaponstate = WEAPON_FIRING;

				// start the animation
				ent->client->anim_priority = ANIM_ATTACK;
                                Coven_AttackAnim (ent);
			}
			else
			{
				if (level.time >= ent->pain_debounce_time)
				{
					gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
					ent->pain_debounce_time = level.time + 1;
				}
				NoAmmoWeaponChange (ent);
			}
		}
		else
		{
			if (ent->client->ps.gunframe == FRAME_IDLE_LAST)
			{
				ent->client->ps.gunframe = FRAME_IDLE_FIRST;
				return;
			}

			if (pause_frames)
			{
				for (n = 0; pause_frames[n]; n++)
				{
					if (ent->client->ps.gunframe == pause_frames[n])
					{
						if (rand()&15)
							return;
					}
				}
			}

			ent->client->ps.gunframe++;
			return;
		}
	}

	if (ent->client->weaponstate == WEAPON_FIRING)
	{
		for (n = 0; fire_frames[n]; n++)
		{
			if (ent->client->ps.gunframe == fire_frames[n])
			{
                                // Quad/tech sounds played elsewhere...
				fire (ent);
				break;
			}
		}

		if (!fire_frames[n])
			ent->client->ps.gunframe++;

		if (ent->client->ps.gunframe == FRAME_IDLE_FIRST+1)
			ent->client->weaponstate = WEAPON_READY;
	}
}

void Weapon_Generic (edict_t *ent, int FRAME_ACTIVATE_LAST, int FRAME_FIRE_LAST, int FRAME_IDLE_LAST, int FRAME_DEACTIVATE_LAST, int *pause_frames, int *fire_frames, void (*fire)(edict_t *ent))
{
	int oldstate = ent->client->weaponstate;

        // Don't freeze weapons if battlesuit.
        if (!ent->battlesuit && (ent->deadflag || ent->s.modelindex != 255))
        // VWep animations screw up corpses
		return;

	Weapon_Generic2 (ent, FRAME_ACTIVATE_LAST, FRAME_FIRE_LAST, 
		FRAME_IDLE_LAST, FRAME_DEACTIVATE_LAST, pause_frames, 
		fire_frames, fire);

	// run the weapon frame again if hasted
        if (CTFApplyHaste (ent) && (oldstate == ent->client->weaponstate))
        {
                // Frame interpolation warps some v_models during certain
                // frames if hasted, so don't haste the following.
                if (!Q_stricmp(ent->client->pers.weapon->pickup_name, "Minigun"))
                {       if ((ent->client->ps.gunframe >= 30) && (ent->client->ps.gunframe <= 35))
                                return;
                }
                if (!Q_stricmp(ent->client->pers.weapon->pickup_name, "Carver"))
                {       if ((ent->client->ps.gunframe >= 9) && (ent->client->ps.gunframe <= 21))
                                return;
                }

		Weapon_Generic2 (ent, FRAME_ACTIVATE_LAST, FRAME_FIRE_LAST,
                        FRAME_IDLE_LAST, FRAME_DEACTIVATE_LAST, pause_frames,
			fire_frames, fire);
	}
}
// COVEN


// COVEN:  ArmourBack weapons!!!
//======================================================================


/*-----/ LIGHT SUIT WEAPONS /-----*/

/*=============================/  Grapple  /=============================*/

void Coven_Hook
(edict_t *ent, int damage, int hold_damage, int speed, int hook_range, int cable_range)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
        vec3_t  base_offset = {8, 8, -8};

        if ((int)dmflags->value & DF_INFINITE_AMMO)
        {       damage *= 2;
                hold_damage *= 2;
        }
	if (is_quad)
        {       damage *= 4;
                hold_damage *= 4;
        }
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
        VectorCopy (base_offset, offset);
        offset[2] += ent->viewheight;
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        Coven_CrosshairVector (forward, start, ent);
        Coven_FireGrapple (ent, start, forward, base_offset, damage, hold_damage, speed, hook_range, cable_range);

        gi.sound (ent, CHAN_WEAPON, gi.soundindex("weapons/hook/hookfire.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

// 0-8 / 9-16 / 17-18 / 19-22
void Weapon_Grapple2 (edict_t *ent)
{
/* Check if player is putting away the grapple. */
	if (ent->client->weaponstate == WEAPON_DROPPING)
        {       if (ent->client->ps.gunframe == 22)
		{       ChangeWeapon (ent);
			return;
		}
		ent->client->ps.gunframe++;
		return;
	}

/* Check if player is pulling out the grapple. */
	if (ent->client->weaponstate == WEAPON_ACTIVATING)
        {
                if (ent->hook)
                {       ent->client->weaponstate = WEAPON_FIRING;
                        ent->client->ps.gunframe = 11;
			return;
		}
                if ( (ent->client->ps.gunframe == 8) ||
                     ((int)(dmflags->value) & DF_FAST_SWITCH) )
		{       ent->client->weaponstate = WEAPON_READY;
                        ent->client->ps.gunframe = 17;
			return;
		}
		ent->client->ps.gunframe++;
		return;
	}

/* Check if player wants (or is forced) to put away the grapple. */
	if ((ent->client->newweapon) && (ent->client->weaponstate != WEAPON_FIRING))
        {       if ((int)(dmflags->value) & DF_FAST_SWITCH)
		{       ChangeWeapon (ent);
			return;
		}
                ent->client->weaponstate = WEAPON_DROPPING;
                ent->client->ps.gunframe = 19;
		return;
	}

/* Check if grapple is ready and idle. */
	if (ent->client->weaponstate == WEAPON_READY)
	{
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{       ent->client->latched_buttons &= ~BUTTON_ATTACK;
                        if (Coven_NoFire ())
                                return;
                        ent->client->ps.gunframe = 9;
                        ent->client->weaponstate = WEAPON_FIRING;

                        /* Start animation. */
                        ent->client->anim_priority = ANIM_ATTACK;
                        Coven_AttackAnim (ent);
			return;
		}

                // Animate idle frames
                ent->client->ps.gunframe = 17;
		return;
	}

/* Check if grapple is firing. */
	if (ent->client->weaponstate == WEAPON_FIRING)
	{
                if (ent->client->ps.gunframe == 9)
                {       Coven_Hook (ent, 40, 4, 1000, 1200, 1500);
                        ent->client->ps.gunframe++;
                        return;
                }

                if (ent->client->ps.gunframe == 11)
                {
                        if (ent->hook)
                        {
                                if (ent->client->newweapon)
                                        ChangeWeapon (ent);
                                return;
                        }
                }

		ent->client->ps.gunframe++;

                if (ent->client->ps.gunframe == 17)
			ent->client->weaponstate = WEAPON_READY;
	}
}

void Weapon_Grapple (edict_t *ent)
{
	int oldstate = ent->client->weaponstate;

        // Don't freeze weapons if battlesuit.
        if (!ent->battlesuit && (ent->deadflag || ent->s.modelindex != 255))
        // VWep animations screw up corpses
		return;

        Weapon_Grapple2 (ent);

	// run the weapon frame again if hasted
        if (CTFApplyHaste (ent) && (oldstate == ent->client->weaponstate))
                Weapon_Grapple2 (ent);
}


/*=========================/  Assault Shotgun  /=========================*/

void Coven_Shotgun (edict_t *ent)
{
        vec3_t  start;
        vec3_t  forward, right;
        vec3_t  offset;
        int     damage = 8;
        int     kick = 12;

        if (ent->client->ps.gunframe == 14)
        {
                ent->client->machinegun_shots = 0;
                if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
                {       ent->client->ps.gunframe = 21;
                        NoAmmoWeaponChange (ent);
                        return;
                }

                if (is_silenced)
                        gi.sound(ent, CHAN_WEAPON, gi.soundindex("weapons/sshotr1b.wav"), 0.2, ATTN_NORM, 0);
                else
                        gi.sound(ent, CHAN_WEAPON, gi.soundindex("weapons/sshotr1b.wav"), 1, ATTN_NORM, 0);
                ent->client->ps.gunframe++;
		return;
        }

        // Animate player
        if (!ent->deadflag)
                if (!ent->battlesuit || (ent->s.frame == ent->client->anim_end) ||
                    (ent->client->anim_priority < ANIM_ATTACK))
                {       ent->client->anim_priority = ANIM_ATTACK;
                        Coven_AttackAnim (ent);
                }

        // Fire!
        if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
	{
                ent->client->machinegun_shots = 0;
		if (level.time >= ent->pain_debounce_time)
		{
			gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
                        if ((ent->client->ps.gunframe != 8) && (ent->client->ps.gunframe != 9))
                                ent->pain_debounce_time = level.time + 1;
		}
                ent->client->ps.gunframe++;
		return;
	}

	AngleVectors (ent->client->v_angle, forward, right, NULL);
	VectorScale (forward, -2, ent->client->kick_origin);
        ent->client->kick_angles[0] = -2 - ent->client->machinegun_shots;
        VectorSet(offset, 0, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

        ent->client->machinegun_shots++;

	if (is_quad)
	{       damage *= 4;
		kick *= 4;
	}
        Coven_PowerupSfx (ent);

        fire_shotgun (ent, start, forward, damage, kick, 600, 400, 6, MOD_SHOTGUN);

        // Send muzzle flash.  Use super shotgun sounds.
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
        gi.WriteByte (MZ_SSHOTGUN | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;
	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Shotgun (edict_t *ent)
{
        static int      pause_frames[]  = {24, 0};
        static int      fire_frames[]   = {8, 9, 10, 14, 0};
        Weapon_Generic (ent, 7, 20, 22, 25, pause_frames, fire_frames, Coven_Shotgun);
}


/*==========================/  Particle Beam  /==========================*/

void Coven_ParticleBeam (edict_t *ent)
{
        vec3_t  start;
        vec3_t  forward, right;
        vec3_t  angles;
        vec3_t  offset;
        vec3_t  spread = {5, 5, 0};
        vec3_t  damage = {12, 2, 4};

        if ( (!(ent->client->buttons & BUTTON_ATTACK)) || Coven_NoFire () )
	{
                ent->client->ps.gunframe = 6;
                gi.sound (ent, CHAN_WEAPON, gi.soundindex("weapons/beam/beamdown.wav"), 1, ATTN_NORM, 0);
                if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
                        NoAmmoWeaponChange (ent);
		return;
	}

	if (ent->client->ps.gunframe == 5)
		ent->client->ps.gunframe = 4;
	else
		ent->client->ps.gunframe = 5;

	if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
	{
		ent->client->ps.gunframe = 6;
		if (level.time >= ent->pain_debounce_time)
		{
                        gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/beam/beamdown.wav"), 1, ATTN_NORM, 0);
			ent->pain_debounce_time = level.time + 1;
		}
		NoAmmoWeaponChange (ent);
		return;
	}

        // Ammo use placed here because player may die and lose all ammo.
	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;

	if (is_quad)
        {       VectorScale (damage, 4, damage);
        }
        Coven_PowerupSfx (ent);

	// get start / end positions
	VectorAdd (ent->client->v_angle, ent->client->kick_angles, angles);
	AngleVectors (angles, forward, right, NULL);
	VectorSet(offset, 0, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_FireBeam (ent, start, forward, spread, damage, 2048);

	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
        gi.WriteByte (MZ_BLASTER | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	PlayerNoise(ent, start, PNOISE_WEAPON);

        // Animate player.
        if (!ent->deadflag)
        {       if (ent->battlesuit)
                {       if ((ent->s.frame == ent->client->anim_end) || (ent->client->anim_priority < ANIM_ATTACK))
                        {       ent->client->anim_priority = ANIM_ATTACK;
                                Coven_AttackAnim (ent);
                        }
                }
                else
                {       ent->client->anim_priority = ANIM_ATTACK;
                        if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
                        {       ent->s.frame = FRAME_crattak1 - (int) (random()+0.25);
                                ent->client->anim_end = FRAME_crattak9;
                        }
                        else
                        {       ent->s.frame = FRAME_attack1 - (int) (random()+0.25);
                                ent->client->anim_end = FRAME_attack8;
                        }
                }
        }
}

void Weapon_ParticleBeam (edict_t *ent)
{
        static int      pause_frames[]  = {19, 0};
        static int      fire_frames[]   = {4, 5, 0};

        if ((ent->client->ps.gunframe == 0) && (!ent->client->resp.spectator))
                gi.sound (ent, CHAN_WEAPON, gi.soundindex("weapons/beam/beamup.wav"), 1, ATTN_NORM, 0);

        Weapon_Generic (ent, 3, 18, 20, 25, pause_frames, fire_frames, Coven_ParticleBeam);
}


/*=========================/  Grenade Launcher  /=========================*/

void Coven_Grenade (edict_t *ent)
{
	vec3_t	offset;
	vec3_t	forward, right;
	vec3_t	start;
        int     damage = 120;
        float   radius = 160;

        if (ent->client->ps.gunframe == 11)
        {       if (!ent->client->pers.inventory[ent->client->ammo_index])
                {       ent->client->ps.gunframe = 18;
                        NoAmmoWeaponChange (ent);
                }
                ent->client->ps.gunframe++;
                return;
        }

	if (is_quad)
        {       damage *= 4;
        }
        Coven_PowerupSfx (ent);

	VectorSet(offset, 8, 8, ent->viewheight-8);
	AngleVectors (ent->client->v_angle, forward, right, NULL);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        fire_grenade (ent, start, forward, damage, 1000, 2.5, radius);

	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_GRENADE | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_GrenadeLauncher (edict_t *ent)
{
	static int	pause_frames[]	= {18, 0};
        static int      fire_frames[]   = {7, 11, 0};
        Weapon_Generic (ent, 6, 17, 19, 24, pause_frames, fire_frames, Coven_Grenade);
}


/*-----/ MEDIUM SUIT WEAPONS /-----*/

/*============================/  Prod Rod  /============================*/

void Coven_Poker (edict_t *ent)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
        int     damage = ((rand() % 5) * 10 + 100);
        int     blast_damage = 40;

        if ((int)dmflags->value & DF_INFINITE_AMMO)
        {       damage *= 2;
                blast_damage *= 2;
        }
	if (is_quad)
        {       damage *= 4;
                blast_damage *= 4;
        }
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
        VectorSet(offset, 0, 8, ent->viewheight-8);
        VectorAdd (offset, vec3_origin, offset);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        Coven_FirePoker (ent, start, forward, 120, damage, blast_damage);

        gi.sound (ent, CHAN_WEAPON, gi.soundindex("weapons/poker/pokefire.wav"), 1, ATTN_NORM, 0);

        ent->client->ps.gunframe++;
	PlayerNoise(ent, start, PNOISE_WEAPON);
}

void Weapon_ProdRod (edict_t *ent)
{
        static int      pause_frames[]  = {10, 0};
        static int      fire_frames[]   = {6, 0};
        Weapon_Generic (ent, 4, 9, 11, 14, pause_frames, fire_frames, Coven_Poker);
}


/*==========================/  Photon Cannon  /==========================*/

void Coven_Photon (edict_t *ent)
{
        vec3_t  start;
        vec3_t  forward, right;
        vec3_t  offset;
        vec3_t  storm = {256, 120, 50};
        int     damage = 100;
        int     cell_cost = ent->client->pers.weapon->quantity;

        if (ent->client->pers.inventory[ent->client->ammo_index] < cell_cost)
	{
		if (level.time >= ent->pain_debounce_time)
		{
			gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
			ent->pain_debounce_time = level.time + 1;
		}
                ent->client->ps.gunframe++;
		NoAmmoWeaponChange (ent);
		return;
	}

        // Ammo use placed here because player may die and lose all ammo.
	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= cell_cost;

	if (is_quad)
        {       damage *= 4;
                storm[2] *= 4;
        }
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -3, ent->client->kick_origin);
	ent->client->kick_angles[0] = -3;

        VectorSet(offset, 0, 7, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        VectorCopy (ent->s.origin, offset);
        offset[2] += ent->viewheight;
        Coven_FirePhoton (ent, start, offset, forward, storm, damage, 4);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_RAILGUN | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;
	PlayerNoise(ent, start, PNOISE_WEAPON);

        // Check if we just used up all the juice.
        if (ent->client->pers.inventory[ent->client->ammo_index] < cell_cost)
                NoAmmoWeaponChange (ent);
}

void Weapon_Photon (edict_t *ent)
{
        static int      pause_frames[]  = {19, 0};
        static int      fire_frames[]   = {8, 0};
        Weapon_Generic (ent, 7, 17, 19, 22, pause_frames, fire_frames, Coven_Photon);
}


/*=============================/  Minigun  /=============================*/

void Coven_StopMinigun (edict_t *ent, int newframe)
{
        ent->client->ps.gunframe = newframe;
        ent->client->weapon_sound = 0;
        gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/chngnd1a.wav"), 1, ATTN_IDLE, 0);
        if (!ent->client->pers.inventory[ent->client->ammo_index])
                NoAmmoWeaponChange (ent);
}

void Coven_Minigun (edict_t *ent)
{
        int     i;
        int     shots = 0;
        vec3_t  start;
        vec3_t  forward, right, up;
        float   r, u;
        vec3_t  offset;
        int     damage = 6;
        int     kick = 2;

	if (ent->client->ps.gunframe == 5)
        {       gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/chngnu1a.wav"), 1, ATTN_IDLE, 0);
                ent->client->machinegun_shots = 0;
        }

        if ( (ent->client->machinegun_shots < 2) &&
             (!(ent->client->buttons & BUTTON_ATTACK) ||
              !ent->client->pers.inventory[ent->client->ammo_index] ||
              Coven_NoFire ()) )
	{
                if (ent->client->ps.gunframe == 9)
                {       Coven_StopMinigun (ent, 32);  return;   }
                if (ent->client->ps.gunframe == 13)
                {       Coven_StopMinigun (ent, 33);  return;   }
                if (ent->client->ps.gunframe == 17)
                {       Coven_StopMinigun (ent, 34);  return;   }
                if (ent->client->ps.gunframe == 21)
                {       Coven_StopMinigun (ent, 35);  return;   }
                if (ent->client->ps.gunframe == 25)
                {       Coven_StopMinigun (ent, 30);  return;   }
                if (ent->client->ps.gunframe == 29)
                {       Coven_StopMinigun (ent, 31);  return;   }
	}
        if (ent->client->ps.gunframe == 29)
                ent->client->ps.gunframe = 6;
	else
		ent->client->ps.gunframe++;

        ent->client->weapon_sound = gi.soundindex("weapons/chngnl1a.wav");

        // Animate player.
        if (!ent->deadflag)
        {       if (ent->battlesuit)
                {       if ((ent->s.frame == ent->client->anim_end) || (ent->client->anim_priority < ANIM_ATTACK))
                        {       ent->client->anim_priority = ANIM_ATTACK;
                                Coven_AttackAnim (ent);
                        }
                }
                else
                {       ent->client->anim_priority = ANIM_ATTACK;
                        if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
                        {       ent->s.frame = FRAME_crattak1 - (ent->client->ps.gunframe & 1);
                                ent->client->anim_end = FRAME_crattak9;
                        }
                        else
                        {       ent->s.frame = FRAME_attack1 - (ent->client->ps.gunframe & 1);
                                ent->client->anim_end = FRAME_attack8;
                        }
                }
        }

        // Fire!
        if ((ent->client->buttons & BUTTON_ATTACK) && (!Coven_NoFire ()))
        {       if (ent->client->machinegun_shots < 3)
                        ent->client->machinegun_shots++;
        }
        else
        {       if (ent->client->machinegun_shots > 1)
                        ent->client->machinegun_shots--;
        }

        shots = ent->client->machinegun_shots;

	if (ent->client->pers.inventory[ent->client->ammo_index] < shots)
		shots = ent->client->pers.inventory[ent->client->ammo_index];

	if (!shots)
        {       ent->client->machinegun_shots = 0;
                if (level.time >= ent->pain_debounce_time)
		{       gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
                        ent->pain_debounce_time = level.time + 1;
		}
		return;
	}

	if (is_quad)
	{       damage *= 4;
		kick *= 4;
	}
        Coven_PowerupSfx (ent);

	for (i=0 ; i<3 ; i++)
	{
		ent->client->kick_origin[i] = crandom() * 0.35;
		ent->client->kick_angles[i] = crandom() * 0.7;
	}

	for (i=0 ; i<shots ; i++)
	{
		// get start / end positions
		AngleVectors (ent->client->v_angle, forward, right, up);
		r = 7 + crandom()*4;
		u = crandom()*4;
		VectorSet(offset, 0, r, u + ent->viewheight-8);
		P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

                fire_bullet (ent, start, forward, damage, kick, 400, 400, MOD_MINIGUN);
	}

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte ((MZ_CHAINGUN1 + shots - 1) | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index] -= shots;
}

void Weapon_Minigun (edict_t *ent)
{
        static int      pause_frames[]  = {36, 0};
        static int      fire_frames[]   = {5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 0};
        Weapon_Generic (ent, 4, 35, 37, 40, pause_frames, fire_frames, Coven_Minigun);
}


/*=============================/  Bazooka  /=============================*/

void Coven_Bazooka (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
        int     damage;
        int     radius_damage;
	float	damage_radius;

        if (ent->client->ps.gunframe == 9)
        {       if (!ent->client->pers.inventory[ent->client->ammo_index])
                {       ent->client->ps.gunframe = 13;
                        NoAmmoWeaponChange (ent);
                }
                ent->client->ps.gunframe++;
                return;
        }

        damage = 100 + (int)(random() * 20.0);
        radius_damage = 120;
        damage_radius = 160;
	if (is_quad)
	{       damage *= 4;
		radius_damage *= 4;
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);
        Coven_CrosshairVector (forward, start, ent);
        Coven_FireRocket (ent, start, forward, damage, 700, damage_radius, radius_damage);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_ROCKET | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Bazooka (edict_t *ent)
{
	static int	pause_frames[]	= {13, 0};
        static int      fire_frames[]   = {5, 9, 0};
        Weapon_Generic (ent, 4, 12, 14, 18, pause_frames, fire_frames, Coven_Bazooka);
}


/*-----/ HEAVY SUIT WEAPONS /-----*/

/*==============================/  Carver  /==============================*/

void Coven_Carver (edict_t *ent)
{
        vec3_t   start;
        vec3_t   forward, right, up;
        vec3_t   offset;
        int      damage = 20;
        qboolean kill = false;

        if (ent->client->ps.gunframe == 4)
                gi.sound(ent, CHAN_AUTO, gi.soundindex("weapons/carver/sawup.wav"), 1, ATTN_IDLE, 0);

        if ((ent->client->ps.gunframe == 8) && ((!(ent->client->buttons & BUTTON_ATTACK)) || Coven_NoFire ()))
	{       ent->client->ps.gunframe = 22;
		ent->client->weapon_sound = 0;
	}
        else if ((ent->client->ps.gunframe == 21) && (ent->client->buttons & BUTTON_ATTACK) && (!Coven_NoFire ()))
                ent->client->ps.gunframe = 9;
	else
		ent->client->ps.gunframe++;

        if (ent->client->ps.gunframe == 22)
	{       ent->client->weapon_sound = 0;
                gi.sound (ent, CHAN_AUTO, gi.soundindex("weapons/carver/sawdown.wav"), 1, ATTN_IDLE, 0);
	}
        else if (ent->client->ps.gunframe == 8)
                ent->client->weapon_sound = gi.soundindex("weapons/carver/sawrun.wav");

        if ((int)dmflags->value & DF_INFINITE_AMMO)
        {       // Carver is always fatal with one touch in inf. ammo mode.
                damage *= 2;
                kill = true;
        }
	if (is_quad)
                damage *= 4;
        Coven_PowerupSfx (ent);

        if ((ent->client->ps.gunframe >= 9) && (ent->client->ps.gunframe <= 21))
        {       kill = true;
                if (CTFApplyHaste (ent))
                        damage *= 4;    // Frames aren't hasted, so double damage.
                else
                        damage *= 2;
        }

        // get start / end positions
        AngleVectors (ent->client->v_angle, forward, right, up);
        VectorSet(offset, 0, 8, ent->viewheight-8);
        P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

        Coven_FireSaw (ent, start, forward, damage, 90, kill);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

void Weapon_Carver (edict_t *ent)
{
        static int      pause_frames[]  = {30, 0};
        static int      fire_frames[]   = {4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 0};
        Weapon_Generic (ent, 3, 29, 31, 34, pause_frames, fire_frames, Coven_Carver);
}


/*===========================/  Flamethrower  /===========================*/

void Coven_FireFire (edict_t *ent, vec3_t g_offset)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
        vec3_t  burn_damage = {6, 9, 50};
        vec3_t  radius_damage = {4, 2, 25};
        vec3_t  spread = {10, 20, 0};
        int     speed;

	if (is_quad)
        {       VectorScale (burn_damage, 4, burn_damage);
                VectorScale (radius_damage, 4, radius_damage);
        }
        Coven_PowerupSfx (ent);

        speed = 500 + (ent->client->machinegun_shots * 100);
        if (speed > 1000)
                speed = 1000;

	AngleVectors (ent->client->v_angle, forward, right, NULL);
        VectorSet(offset, 10, 8, ent->viewheight-8);
	VectorAdd (offset, g_offset, offset);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

        Coven_FireFlameThrower (ent, start, spread, forward, speed, burn_damage, radius_damage, 10, 0, MOD_FLAMETHROWER, MOD_FLAMETHROWER);
        Coven_FireFlameThrower (ent, start, spread, forward, speed*2, burn_damage, radius_damage, 10, 0, MOD_FLAMETHROWER, MOD_FLAMETHROWER);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
        gi.WriteByte (MZ_HYPERBLASTER | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

void Coven_Flamethrower (edict_t *ent)
{
        if (ent->client->ps.gunframe == 7)
                ent->client->machinegun_shots = 1;
        else
                ent->client->machinegun_shots++;

        if (ent->client->pers.inventory[ent->client->ammo_index])
        {
                Coven_FireFire (ent, vec3_origin);

                if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                        ent->client->pers.inventory[ent->client->ammo_index]--;

                // Animate player
                if (!ent->deadflag)
                        if (!ent->battlesuit || (ent->s.frame == ent->client->anim_end) ||
                            (ent->client->anim_priority < ANIM_ATTACK))
                        {       ent->client->anim_priority = ANIM_ATTACK;
                                Coven_AttackAnim (ent);
                        }
        }

        ent->client->ps.gunframe++;

        // Check if we just used up all the fuel.
        if (ent->client->ps.gunframe == 12)
                if (! ent->client->pers.inventory[ent->client->ammo_index] )
                        NoAmmoWeaponChange (ent);
}

void Weapon_Flamethrower (edict_t *ent)
{
        static int      pause_frames[]  = {18, 0};
        static int      fire_frames[]   = {7, 8, 9, 10, 11, 0};
        Weapon_Generic (ent, 6, 17, 19, 23, pause_frames, fire_frames, Coven_Flamethrower);
}


/*===========================/  Auto Shotgun  /===========================*/

void Coven_AutoShotgun (edict_t *ent)
{
        vec3_t  start;
        vec3_t  forward, right;
        vec3_t  offset;
        int     damage = 8;
        int     kick = 12;

        if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
	{
		if (level.time >= ent->pain_debounce_time)
		{
                        if ( (ent->client->ps.gunframe != 8) &&
                             (ent->client->ps.gunframe != 11) &&
                             (ent->client->ps.gunframe != 14) )
                        {
                                gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
                                if ((ent->client->ps.gunframe != 7) && (ent->client->ps.gunframe != 10))
                                        ent->pain_debounce_time = level.time + 1;
                        }
		}
                ent->client->ps.gunframe++;
                if (ent->client->ps.gunframe == 15)
                        NoAmmoWeaponChange (ent);
		return;
	}

        if ( (ent->client->ps.gunframe == 8) ||
             (ent->client->ps.gunframe == 11) ||
             (ent->client->ps.gunframe == 14) )
        {
                if ( (!(ent->client->buttons & BUTTON_ATTACK)) || Coven_NoFire () )
                {       ent->client->ps.gunframe++;
                        return;
                }
        }

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -2;

        // Make slight triangle shot pattern.
        if ((ent->client->ps.gunframe == 7) || (ent->client->ps.gunframe == 8))
                VectorSet(offset, 0, 4, ent->viewheight-3);
        else if ((ent->client->ps.gunframe == 10) || (ent->client->ps.gunframe == 11))
                VectorSet(offset, 0, 12, ent->viewheight-8);
        else if ((ent->client->ps.gunframe == 13) || (ent->client->ps.gunframe == 14))
                VectorSet(offset, 0, 4, ent->viewheight-13);
        else
                VectorSet(offset, 0, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	if (is_quad)
	{
		damage *= 4;
		kick *= 4;
	}
        Coven_PowerupSfx (ent);

        fire_shotgun (ent, start, forward, damage, kick, 600, 400, 6, MOD_AUTOSHOTGUN);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
	gi.WriteByte (MZ_SSHOTGUN | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;
	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index]--;

        // Animate player
        if (!ent->deadflag)
                if (!ent->battlesuit || (ent->s.frame == ent->client->anim_end) ||
                    (ent->client->anim_priority < ANIM_ATTACK))
                {       ent->client->anim_priority = ANIM_ATTACK;
                        Coven_AttackAnim (ent);
                }

        // Check if we just used up all the shells.
        if (ent->client->ps.gunframe == 15)
                if (ent->client->pers.inventory[ent->client->ammo_index] < 1)
                        NoAmmoWeaponChange (ent);
}

void Weapon_AutoShotgun (edict_t *ent)
{
        static int      pause_frames[]  = {16, 0};
        static int      fire_frames[]   = {7, 8, 10, 11, 13, 14, 0};
        Weapon_Generic (ent, 6, 15, 17, 21, pause_frames, fire_frames, Coven_AutoShotgun);
}


/*===============================/  MIRV  /===============================*/

void Coven_MIRV (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
        int     damage;
        int     radius_damage;
	float	damage_radius;
        vec3_t  homer_damage;
        edict_t *targ;

        if (ent->client->ps.gunframe == 12)
        {
                if (ent->client->pers.inventory[ent->client->ammo_index] < 5)
                {       ent->client->ps.gunframe = 25;
                        NoAmmoWeaponChange (ent);
                        return;
                }

                ent->client->ps.gunframe++;
		return;
        }

        if (ent->client->ps.gunframe == 13)
        {
                if (is_silenced)
                        gi.sound(ent, CHAN_WEAPON, gi.soundindex("weapons/mirv/mirvload.wav"), 0.2, ATTN_NORM, 0);
                else
                        gi.sound(ent, CHAN_WEAPON, gi.soundindex("weapons/mirv/mirvload.wav"), 1, ATTN_NORM, 0);
                ent->client->ps.gunframe++;
		return;
        }

        if (ent->client->pers.inventory[ent->client->ammo_index] < 5)
	{
		if (level.time >= ent->pain_debounce_time)
		{
			gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
			ent->pain_debounce_time = level.time + 1;
		}
                ent->client->ps.gunframe++;
		NoAmmoWeaponChange (ent);
		return;
	}

        damage = 140 + (int)(random() * 20.0);
        radius_damage = 160;
        damage_radius = 200;
        VectorSet (homer_damage, 60, 20, 80);
	if (is_quad)
	{       damage *= 4;
		radius_damage *= 4;
                VectorScale (homer_damage, 4, homer_damage);
	}
        Coven_PowerupSfx (ent);

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);

	// make a big pitch kick with an inverse fall
        ent->client->v_dmg_pitch = -30;
	ent->client->v_dmg_roll = crandom()*8;
	ent->client->v_dmg_time = level.time + DAMAGE_TIME;

        // Check for a target directly in front of player, and track it.
        VectorCopy (ent->s.origin, start);
        start[2] += ent->viewheight;
        targ = Coven_MIRV_Target (ent, start, forward);

        // Find gun source.
	VectorSet(offset, 8, 8, ent->viewheight-8);
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

        // If we don't have a target, check for one in front of gun.
        if (!targ)
                targ = Coven_MIRV_Target (ent, start, forward);

        Coven_CrosshairVector (forward, start, ent);
        Coven_FireMIRV (ent, start, forward, damage, 500, damage_radius, radius_damage, 4, homer_damage, targ);

	// send muzzle flash
	gi.WriteByte (svc_muzzleflash);
	gi.WriteShort (ent-g_edicts);
        gi.WriteByte (MZ_BFG | is_silenced);
	gi.multicast (ent->s.origin, MULTICAST_PVS);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
                ent->client->pers.inventory[ent->client->ammo_index] -= 5;
}

void Weapon_MIRV (edict_t *ent)
{
        static int      pause_frames[]  = {25, 0};
        static int      fire_frames[]   = {5, 12, 13, 0};
        Weapon_Generic (ent, 4, 24, 26, 30, pause_frames, fire_frames, Coven_MIRV);
}


//======================================================================
// COVEN
