/*
**  graph class
**
**  (c) 1997 mike warren
**  mikeBot
**
**
**  builds a directed graph of the current level as the bot moves through
**  the level. as position is updates, this class figures out where the bot
**  is and if that new node has a floor polygon or is lava/slime/water, it
**  is added to the graph.  use the public: interface functions to get info
**  about the level. Provides line of sight, leaf type, etc
**
*/



#include "graph.h"
#include "stack.h"
#include <memory.h>
#include <string.h>


/*
**  newMap : loads new map, if possible
**
*/

int bspGraph::newMap( char * fname, int )
{
  if( map )
    {
      save();			// save current directed graph
      delete map;		// erase bspFile
    }

  printf("loading BSP...");
  fflush( stdout );

  try {
    map = new bspFile( fname );
  }
  catch( char * err )
    {
      fprintf(stderr, "\n%s\nbspGraph::newMap(): level not loaded\n",err);
      delete map;
      map=0;
      return FALSE;
    }

  eraseGraph();

//  int links;
  int i;
  vector tv;

				// allocate space for graph

  listSize = map->leaves->getNum();
  adjList = new adjNode*[ listSize+1 ];
  for( i=0; i <= listSize; i++ )
    if( !(adjList[ i ] = new adjNode(-1, tv)) )
      fprintf(stderr, "bspGraph::newMap(): node failed\n");;
  t = new table[ listSize+1 ];
  totalLinks=0;

  printf("%d leaves\n", listSize );

  strcpy( gfname, fname );
  i=strlen( gfname );
  gfname[i-3]='g';
  gfname[i-2]='n';
  gfname[i-1]=0;

  printf("loading directed graph from `%s'...", gfname);
  fflush( stdout );
	int   links=0;

				// 
				//  TODO : endianess; use mFile
/*				// 

  int fd = open( gfname, O_RDONLY );
  if( fd < 2 )
    perror("bspGraph::newMap(): open");
  else
    {
      read( fd, (char *)&i, 4 );
      for( links=0; links < i; links++ )
	{
	  read( fd, (char *)&s, 4 );
	  read( fd, (char *)&e, 4 );

	  read( fd, (char *)&f, 4 );
	  tv.setx(f);
	  read( fd, (char *)&f, 4 );
	  tv.sety(f);
	  read( fd, (char *)&f, 4 );
	  tv.setz(f);
	  addLink( s, e, &tv );
	}
      close( fd );
    }
*/      

  printf("%d links made\n", links);

//  preprocessGraph();

  return TRUE;

}

/*
**  updatePosition : call this everytime the bot moves...figures out where
**		     it is and possibly starts remaking paths
**
*/

void bspGraph::updatePosition( vector newPos )
{
  oldNode = currentNode;
  currentNode = findLeaf( newPos );

  oldPosition = position;
  position=newPos;

  if( !dead && currentNode && oldNode && !pointOverLava( newPos ) )
    {
	if ( currentNode != oldNode )
	  if( !isLineBlocked( getLeafOrigin(oldNode), getLeafOrigin(currentNode),0 ) )
	     {
	       vector tv = getLeafOrigin( oldNode );
	       vector tv2 = getLeafOrigin( currentNode );
	       tv += tv2;
	       tv /= (float)2.0;
	    
	       if( addLink( oldNode, currentNode, &tv ) )
#if DEBUG & MBM
		 printf("bspGraph::updatePosition(): link added %d --> %d\n", oldNode, currentNode );
#else
	       ;
#endif
	     }
    }

  if( dead )
    dead=0;

}


/*
**  getLeafOrigin : return the center of the bbox for the specified leaf
**
*/

vector bspGraph::getLeafOrigin( int x )
{
  vector r;
  bspLeaf * l;

  if( !map )
    return r;

  l = map->leaves->getLeaf( x );
  if( !l )
    return r;

  r = l->getBboxMin();
  vector r2 = l->getBboxMax();
  r += r2;
  r /= (float)2.0;

  r.setz( position.getz() );

  return r;
}
  

/*
**  getCurrentLeafType : which type of leaf is the bot in
**
*/

int bspGraph::getCurrentLeafType()
{
  if( !map )
    return -100;

  return map->leaves->getLeaf( currentNode )->getType();

}

/*
**  getLeafType : same as above, but uses vector to find leaf first
**
*/

int bspGraph::getLeafType( vector pos )
{
  if( !map )
    return -100;

  return map->leaves->getLeaf( findLeaf(pos) )->getType();
}

/*
**  isShotBlocked : return TRUE if there is a wall between bots current pos
**                  and the one passed in
**
*/

int bspGraph::isShotBlocked( vector origin, vector * hp )
{

  if( isLineBlocked( position, origin, hp,FALSE ) )
    return TRUE;
  else
    return FALSE;

}


/*
**  findLeaf : determines which leaf the pos vector is in
**
*/

int bspGraph::findLeaf( vector origin )
{

  int node;
  int isNode = 1;
  float dist;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while( 1 )
    {
      if( isNode )
	{
	  dist = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) *
	    (origin)) - 
	      (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  if( dist > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      node = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      node = map->nodes->getNode(node)->getBack();
	    }
	}
      else
	break;
    }

#if DEBUG & MBM
//  printf("bspGraph::findLeaf(): leaf %d\n", node);
#endif

  return node;
}
	  


/*
**  isLineBlocked : returns 0 if the line is not blocked, or the index
**	            of the leaf which blocks it.
**
*/

int bspGraph::isLineBlocked( vector sv, vector ev, vector * hitPoint, int lava)
{

  stack< vector > st;
  stack< int > st1;
  vector start=sv, end=ev, tv;
  float s,e;
  int node, startNode, endNode;
  int isNode=1;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while ( 1 )
    {
      if( node == -1 )
	break;

      if( isNode )
	{
	  // e = distance from end to plane
	  // s = distance from start to plane

	  e = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (end)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  s = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (start)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
      
#define RE 0.001

	  if( e > 0.0 )
	    endNode = map->nodes->getNode(node)->getFront();
	  else
	    endNode = map->nodes->getNode(node)->getBack();

	  if( s > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      startNode = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      startNode = map->nodes->getNode(node)->getBack();
	    }

	  if( (s < -RE && e > RE) || (s > RE && e < -RE) )
	    {
	      st.push( end );
	      st1.push( endNode );
	      tv = start;
	      tv *= e;
	      end *= s;
	      end = end - tv;
	      end /= (s-e);
	    }
	  node = startNode;
	}
      else
	{
	  if( !lava )
	    {
	      if( map->leaves->getLeaf( node )->getType() == bspLeaf::solid )
		{
		  if( hitPoint )
		    *hitPoint = end;
		  return map->leaves->getLeaf(node)->getType();
		}
	    }
	  else
	    {
	      if( map->leaves->getLeaf( node )->getType() != bspLeaf::normal )
		{
		  if( hitPoint )
		    *hitPoint = end;
		  return map->leaves->getLeaf(node)->getType();
		}
	    }		

	  start = end;
	  if( st.isEmpty() )
	    return FALSE;

	  end = st.pop();
	  node = st1.pop();
	  isNode=1;
	}
    }


  return FALSE;
  
}

/*
**  lineHitsLava : returns TRUE if the line passes throgh lava
**
*/

int bspGraph::lineHitsLava( vector sv, vector ev )
{

  stack< vector > st;
  stack< int > st1;
  vector start=sv, end=ev, tv;
  float s,e;
  int node, startNode, endNode;
  int isNode=1;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while ( 1 )
    {
      if( node == -1 )
	break;

      if( isNode )
	{
	  // e = distance from end to plane
	  // s = distance from start to plane

	  e = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (end)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  s = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (start)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
      
	  if( e > 0.0 )
	    endNode = map->nodes->getNode(node)->getFront();
	  else
	    endNode = map->nodes->getNode(node)->getBack();

	  if( s > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      startNode = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      startNode = map->nodes->getNode(node)->getBack();
	    }

	  if( (s < -RE && e > RE) || (s > RE && e < -RE) )
	    {
	      st.push( end );
	      st1.push( endNode );
	      tv = start;
	      tv *= e;
	      end *= s;
	      end = end - tv;
	      end /= (s-e);
	    }
	  node = startNode;
	}
      else
	{
	  if( map->leaves->getLeaf( node )->getType() == bspLeaf::lava )
	    return TRUE;

	  start = end;
	  if( st.isEmpty() )
	    return FALSE;

	  end = st.pop();
	  node = st1.pop();
	  isNode=1;
	}
    }

  return FALSE;
  
}


/*
**  updateTable : finds all paths from leaf a
**
*/

void bspGraph::updateTable( int a )
{
  static queue q;
  int v, w;
  adjNode * p;

#if DEBUG & MBM
  if( a == 0 )
    printf("bspGraph::updateTable(): start node 0\n");
#endif

  for( int i=0; i < listSize; i++ )
    {
      t[i].p=0;
      t[i].dist=-1;
    }

  t[ a ].dist = 0;		// start node
  q.add( a );

  while( !q.isEmpty() )
    {
      v = q.get();
#if DEBUG & MBM
      if( v==0 )
	printf("bspGraph::updateTable(): WARNING: v==0\n");
#endif
      p = adjList[ v ];
      while( p->next )
	{
	  w = p->next->data;
	  p = p->next;
	  if( t[w].dist == -1 )
	    {
	      t[w].dist = t[v].dist + 1;
	      t[w].p = v;
	      q.add( w );
	    }
	}
    }

}

/*
**  removeLink : takes out link from A to B
**
*/

int bspGraph::removeLink( int a, int b )
{

 if( a > listSize || b > listSize )
    return FALSE;

 adjNode * p = adjList[ a ];
 adjNode * q;

 while( p->next )
   {
     if( p->next->data == b )
       {
	 q = p->next->next;
	 delete p->next;
	 p->next = q;
	 return TRUE;
       }
     p = p->next;
   }

 return FALSE;

}

/*
**  isLink : returns TRUE iff A --> B is a link
**
*/

int bspGraph::isLink( int a, int b )
{
  if( a >= listSize || b >= listSize )
    return TRUE;
  if( a < 0 || b < 0 )
    return TRUE;

  adjNode * p = adjList[ a ];

  if( !p )
    {
      fprintf(stderr, "bspGraph::isLink(): problem\n");
      return FALSE;
    }

  while( p->next )
    {
      if( p->next->data == b )
	return TRUE;
      p = p->next;
    }

  return FALSE;

}

/*
**  addLink : links node A to B
**
*/

int bspGraph::addLink( int a, int b, vector * origin )
{
  if( a > listSize || b > listSize )
    return FALSE;
  
  if( isLink( a,b ) )
    return FALSE;

  adjNode * p = adjList[ a ];

  if( !p )
    {
      fprintf(stderr,"bspGraph::addLink(): problem\n");
      return FALSE;
    }

#if DEBUG & MBM
  int i=0;
#endif
  
  while( p->next )
    {
      p = p->next;
#if DEBUG & MBM
      i++;
#endif
    }

#if DEBUG & MBM
  printf("%d links from node %d ", i, a );
  if( !origin )
    oldPosition.print();
  else
    origin->print();
  printf("\n");
#endif

  if( !origin )
    p->next = new adjNode( b, oldPosition );
  else
    p->next = new adjNode( b, *origin );
  totalLinks++;

  return TRUE;

}

/*
**  numLinks : returns number of links out of node
**
*/

int bspGraph::numLinks( int a )
{
  if( a >= listSize )
    return -1;
  if( a < 0 )
    return -1;

  adjNode * p = adjList[ a ];

  if( !p )
    {
      fprintf(stderr, "bspGraph::isLink(): problem\n");
      return -1;
    }

  int i=0;

  while( p->next )
    {
      i++;
      p = p->next;
    }

  return i;

}


/*
**  eraseGraph : deletes directed-graph stuff if it exists
**
*/

void bspGraph::eraseGraph()
{
  if( listSize <= 0 )
    return;
  
  adjNode *p, *q;
  int i;
  
  delete[] t;
  for( i=0; i < listSize; i++ )
    {
      p = adjList[i];
      while( p->next )
	{
	  q = p;
	  p = p->next;
	  delete q;
	}
      adjList[i]=0;
    }
  
  delete[] adjList;

  adjList=0;
  listSize=0;
  t = 0;
  
}

/*
**  save : saves to a file...
**
*/

int bspGraph::save()
{
  if( gfname[0]==0 )
    return 0;

  int fd = open( gfname, O_WRONLY|O_CREAT|O_TRUNC, 0600 );
  vector tv;
  float f;

  if( fd < 2 )
    {
      perror( gfname );
      return 0;
    }

  if( write( fd, (char *)&totalLinks, 4 ) < 4 )
    {
      perror(gfname);
      close( fd );
      return 0;
    }

  adjNode * p;

#if DEBUG & MBM
//  int j=0,k=0;
//  printf("%d links to save\n", totalLinks);
#endif

  printf("saving level graph");
  fflush( stdout );

  for( int i=0; i < listSize; i++ )
    {
      if( !(i%100) )
	{
	  printf(".");
	  fflush(stdout);
	}
      p = adjList[i];
#if DEBUG & MBM
//      k=0;
#endif
      while( p->next )
	{
#if DEBUG & MBM
//	  k++;
#endif
	  if( write( fd, (char *)&i, 4 ) < 4 )
	    {
	      perror(gfname);
	      close( fd );
	      return 0;
	    }
	  if( write( fd, (char *)&p->next->data, 4 ) < 4 )
	    {
	      perror(gfname);
	      close( fd );
	      return 0;
	    }

	  f = p->next->origin.getx();
	  if( write( fd, (char *)&f, 4 ) < 4 )
	    {
	      perror(gfname);
	      close( fd );
	      return 0;
	    }

	  f = p->next->origin.gety();
	  if( write( fd, (char *)&f, 4 ) < 4 )
	    {
	      perror(gfname);
	      close( fd );
	      return 0;
	    }

	  f = p->next->origin.getz();
	  if( write( fd, (char *)&f, 4 ) < 4 )
	    {
	      perror(gfname);
	      close( fd );
	      return 0;
	    }

	  p=p->next;
#if DEBUG & MBM
//	  j++;
#endif
	}
#if DEBUG & MBM
//      printf("node %d : %d links saved\n", i,k );
#endif
    }

  printf("saved %d links\n", totalLinks);

  gfname[0]=0;			// so it won't be saved twice

  return totalLinks;
}

